<?php

namespace Modules\Courier\app\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\Courier\app\Http\Requests\CourierRequest;
use Modules\Courier\app\Http\Requests\CourierOrderRequest;
use Modules\Courier\app\Http\Resources\CourierOrderResource;
use Modules\Courier\app\Http\Resources\CourierOrderStatsResource;
use Modules\Courier\app\Services\CourierService;
use Modules\Courier\app\Services\CourierOrderService;
use Illuminate\Support\Facades\Log;
class CourierController extends Controller
{
    protected CourierService $courierService;
    protected CourierOrderService $courierOrderService;

    public function __construct(
        CourierService $courierService,
        CourierOrderService $courierOrderService
    ) {
        $this->courierService = $courierService;
        $this->courierOrderService = $courierOrderService;
    }

    /**
     * Display a listing of couriers.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $filters = $request->only(['name', 'family', 'mobile', 'national_code', 'branch_id', 'user_id', 'search', 'delivery_percent', 'phone']);
            $perPage = (int) $request->get('per_page', 15);

            $couriers = $this->courierService->getAllCouriers($filters, $perPage);

            return response()->json([
                'status' => 'success',
                'data' => $couriers
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در دریافت لیست پیک‌ها',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('courier::create');
    }

    /**
     * Store a newly created courier.
     */
    public function store(CourierRequest $request): JsonResponse
    {
        try {
            $courier = $this->courierService->createCourier($request->validated());

            return response()->json([
                'status' => 'success',
                'message' => 'پیک با موفقیت ایجاد شد',
                'data' => $courier
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در ایجاد پیک',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified courier.
     */
    public function show(int $id): JsonResponse
    {
        try {
            $courier = $this->courierService->findCourier($id);

            if (!$courier) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'پیک یافت نشد'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'data' => $courier
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در دریافت اطلاعات پیک',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        return view('courier::edit');
    }

    /**
     * Update the specified courier.
     */
    public function update(CourierRequest $request, int $id): JsonResponse
    {
        try {
            $updated = $this->courierService->updateCourier($id, $request->validated());

            if (!$updated) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'خطا در بروزرسانی پیک'
                ], 400);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'پیک با موفقیت بروزرسانی شد'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در بروزرسانی پیک',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified courier.
     */
    public function destroy(int $id): JsonResponse
    {
        try {
            $deleted = $this->courierService->deleteCourier($id);

            if (!$deleted) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'خطا در حذف پیک'
                ], 400);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'پیک با موفقیت حذف شد'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در حذف پیک',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search for users by name, family, or mobile.
     */
    public function searchUsers(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'query' => 'required_without:search|string|min:2',
                'search' => 'required_without:query|string|min:2',
            ]);

            $searchQuery = $request->input('query') ?: $request->input('search');
            $users = $this->courierService->searchUsers(['query' => $searchQuery]);

            return response()->json([
                'status' => 'success',
                'data' => $users
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در جستجوی کاربران',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search for existing user and create courier if user exists.
     * If user doesn't exist, create user first then create courier.
     */
    public function searchAndCreate(CourierRequest $request): JsonResponse
    {
        try {
            $courier = $this->courierService->searchAndCreateCourier($request->validated());

            return response()->json([
                'status' => 'success',
                'message' => 'پیک با موفقیت ایجاد شد',
                'data' => $courier
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Get orders assigned to a specific courier by user_id.
     */
    public function getCourierOrders(CourierOrderRequest $request): JsonResponse
    {
        try {
            $validated = $request->validated();
            $userId = $validated['user_id'];
            $perPage = $validated['per_page'] ?? 15;

            // Validate courier exists
            $this->courierOrderService->validateCourier($userId);
            // Get filters
            $filters = $request->only(['order_status', 'start_date', 'end_date', 'payment_status', 'branch_id']);

            // Get orders
            $orders = $this->courierOrderService->getCourierOrders($userId, $filters, $perPage);

            return response()->json([
                'status' => 'success',
                'message' => 'لیست سفارشات پیک با موفقیت دریافت شد',
                'data' => $orders
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در اعتبارسنجی داده‌ها',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('خطا در دریافت سفارشات پیک: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage(),
                'error' => config('app.debug') ? $e->getMessage() : 'خطا در دریافت سفارشات پیک'
            ], 500);
        }
    }

    /**
     * Get courier order statistics.
     */
    public function getCourierOrderStats(CourierOrderRequest $request): JsonResponse
    {
        try {
            $validated = $request->validated();
            $userId = $validated['user_id'];

            // Validate courier exists
            $this->courierOrderService->validateCourier($userId);

            // Get filters
            $filters = $request->only(['order_status', 'start_date', 'end_date', 'payment_status', 'branch_id']);

            // Get statistics
            $stats = $this->courierOrderService->getCourierOrderStats($userId, $filters);

            // Get courier info
            $courierInfo = $this->courierOrderService->getCourierInfo($userId);

            return response()->json([
                'status' => 'success',
                'message' => 'آمار سفارشات پیک با موفقیت دریافت شد',
                'data' => [
                    'courier' => $courierInfo,
                    'statistics' => new CourierOrderStatsResource($stats)
                ]
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در اعتبارسنجی داده‌ها',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('خطا در دریافت آمار سفارشات پیک: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage(),
                'error' => config('app.debug') ? $e->getMessage() : 'خطا در دریافت آمار سفارشات پیک'
            ], 500);
        }
    }
}
