<?php

namespace Modules\Courier\app\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class BranchCourier extends Model
{
    use SoftDeletes;

    protected $guarded = ['id'];

    protected $dates = ['created_at', 'updated_at', 'deleted_at'];

    protected $casts = [
        'branch_id' => 'integer',
        'user_id' => 'integer',
        'delivery_percent' => 'decimal:2',
        'national_code' => 'string',
    ];

    /**
     * Get the branch that owns the courier.
     */
    public function branch(): BelongsTo
    {
        return $this->belongsTo(\App\Models\Branch::class);
    }

    /**
     * Get the user that owns the courier.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(\App\Models\User::class);
    }

    /**
     * Get the delivery times for the courier.
     */
    public function deliveryTimes(): HasMany
    {
        return $this->hasMany(DeliveryTime::class, 'branch_courier_id');
    }

    /**
     * Scope to filter by branch.
     */
    public function scopeByBranch($query, $branchId)
    {
        return $query->where('branch_id', $branchId);
    }

    /**
     * Scope to filter by user.
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }
}
