<?php

namespace Modules\Courier\app\Repositories;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\DB;
use Modules\Courier\app\Models\BranchCourier;
use Modules\Courier\app\Models\DeliveryTime;
use App\Models\User;

class BranchCourierRepository implements BranchCourierRepositoryInterface
{
    /**
     * Retrieve all couriers with optional filters and pagination.
     * Filters on 'name', 'family', and 'mobile' are applied on the related 'user' model.
     *
     * @param array $filters
     * @param int $perPage
     * @return LengthAwarePaginator
     */
    public function getAll(array $filters = [], int $perPage = 15): LengthAwarePaginator
    {
        $query = BranchCourier::with(['user', 'branch', 'deliveryTimes']);

        // Filter by search query (searches in name, family, and mobile)
        if (!empty($filters['search'])) {
            $searchQuery = $filters['search'];
            $query->where('name', 'like', "%{$searchQuery}%")
                  ->orWhere('family', 'like', "%{$searchQuery}%")
                  ->orWhere('mobile', 'like', "%{$searchQuery}%");
        } else {
            // Individual filters for backward compatibility
            // Filter by user's name
            if (!empty($filters['name'])) {
                $query->where('name', 'like', "%{$filters['name']}%");
                };


            // Filter by user's family
            if (!empty($filters['family'])) {
                $query->where('family', 'like', "%{$filters['family']}%");
                };


            // Filter by user's mobile
            if (!empty($filters['mobile'])) {
                $query->where('mobile', 'like', "%{$filters['mobile']}%");
                };

        }

        // Filter by branch_id on branch_couriers table
        if (!empty($filters['branch_id'])) {
            $query->where('branch_id', $filters['branch_id']);
        }

        // Filter by user_id on branch_couriers table
        if (!empty($filters['user_id'])) {
            $query->where('user_id', $filters['user_id']);
        }

        // Filter by delivery_percent
        if (!empty($filters['delivery_percent'])) {
            $query->where('delivery_percent', $filters['delivery_percent']);
        }

        // Filter by phone
        if (!empty($filters['phone'])) {
            $query->where('phone', 'like', "%{$filters['phone']}%");
        }

        // Filter by national_code
        if (!empty($filters['national_code'])) {
            $query->where('national_code', 'like', "%{$filters['national_code']}%");
        }

        $paginator = $query->paginate($perPage);

        // Append user's fields directly to the result for convenience
        $paginator->getCollection()->transform(function ($item) {
            if ($item) {
                $item->name = $item->name;
                $item->family = $item->family;
                $item->mobile = $item->mobile;
                $item->birth_date = $item->birth_date;
                $item->national_code = $item->national_code;
            } else {
                $item->name = null;
                $item->family = null;
                $item->mobile = null;
                $item->birth_date = null;
                $item->national_code = null;
            }
            return $item;
        });

        return $paginator;
    }

    /**
     * Find a courier by ID along with related models.
     *
     * @param int $id
     * @return BranchCourier|null
     */
    public function findById(int $id): ?BranchCourier
    {
        return BranchCourier::with(['user', 'branch', 'deliveryTimes'])->find($id);
    }

    /**
     * Create a new courier and associate delivery times in a transaction.
     *
     * @param array $data
     * @return BranchCourier
     */
    public function create(array $data): BranchCourier
    {
        $deliveryTimes = $data['delivery_times'] ?? [];
        unset($data['delivery_times']);
        return DB::transaction(function () use ($data, $deliveryTimes) {
            $courier = BranchCourier::create($data);

            foreach ($deliveryTimes as $time) {
                DeliveryTime::create([
                    'branch_courier_id' => $courier->id,
                    'start_time'        => $time['start_time'],
                    'end_time'          => $time['end_time'],
                ]);
            }

            return $courier->load(['user', 'branch', 'deliveryTimes']);
        });
    }

    /**
     * Update courier and replace delivery times inside a transaction.
     *
     * @param int $id
     * @param array $data
     * @return bool
     */
    public function update(int $id, array $data): bool
    {
        $deliveryTimes = $data['delivery_times'] ?? null;
        unset($data['delivery_times']);

        return DB::transaction(function () use ($id, $data, $deliveryTimes) {
            $courier = BranchCourier::findOrFail($id);

            // Cast numeric fields properly
            if (isset($data['delivery_percent'])) {
                $data['delivery_percent'] = (float) $data['delivery_percent'];
            }
            if (isset($data['branch_id'])) {
                $data['branch_id'] = (int) $data['branch_id'];
            }
            if (isset($data['user_id'])) {
                $data['user_id'] = (int) $data['user_id'];
            }

            $updated = $courier->update($data);

            if (is_array($deliveryTimes)) {
                DeliveryTime::where('branch_courier_id', $courier->id)->delete();
                foreach ($deliveryTimes as $time) {
                    DeliveryTime::create([
                        'branch_courier_id' => $courier->id,
                        'start_time'        => $time['start_time'],
                        'end_time'          => $time['end_time'],
                    ]);
                }
            }

            return $updated;
        });
    }

    /**
     * Soft delete a courier by ID.
     *
     * @param int $id
     * @return bool
     */
    public function delete(int $id): bool
    {
        $courier = BranchCourier::findOrFail($id);
        $courier->deliveryTimes()->delete();
        return (bool) $courier->delete();
    }

    /**
     * Search for users by name, family, or mobile.
     *
     * @param array $filters
     * @return array
     */
    public function searchUsers(array $filters): array
    {
        $query = BranchCourier::query();

        if (!empty($filters['query'])) {
            $query->where(function ($q) use ($filters) {
                $q->where('name', 'like', "%{$filters['query']}%")
                  ->orWhere('family', 'like', "%{$filters['query']}%")
                  ->orWhere('mobile', 'like', "%{$filters['query']}%")
                  ->orWhere('national_code', 'like', "%{$filters['query']}%");
            });
        }

        // Support for search parameter as well
        if (!empty($filters['search'])) {
            $query->where(function ($q) use ($filters) {
                $q->where('name', 'like', "%{$filters['search']}%")
                  ->orWhere('family', 'like', "%{$filters['search']}%")
                  ->orWhere('mobile', 'like', "%{$filters['search']}%")
                  ->orWhere('national_code', 'like', "%{$filters['search']}%");
            });
        }

        return $query->select('id', 'name', 'family', 'mobile', 'national_code', 'birth_date')
                    ->limit(10)
                    ->get()
                    ->toArray();
    }

    /**
     * Search for existing user by mobile and create courier if user exists.
     * If user doesn't exist, create user first then create courier.
     *
     * @param array $data
     * @return BranchCourier
     */
    public function searchAndCreate(array $data): BranchCourier
    {
        return DB::transaction(function () use ($data) {
            // Search for existing user by mobile
            $user = User::where('mobile', $data['mobile'])->first();
            if (!$user) {
                // Create new user if not found
                $user = User::create([
                    'name' => $data['name'],
                    'family' => $data['family'],
                    'mobile' => $data['mobile'],
                    'national_code' => $data['national_code'],
                    'email' => $data['mobile'] . '@limoofood.com', // Generate email from mobile
                    'password' => bcrypt('123456'), // Default password
                    'birth_date' => $data['birth_date'] ?? null,
                ]);
            }

            // Check if courier already exists for this user and branch
            $existingCourier = BranchCourier::where('user_id', $user->id)
                ->where('branch_id', $data['branch_id'])
                ->whereNull('deleted_at')
                ->first();

            if ($existingCourier) {
                throw new \Exception('Courier already exists for this user and branch.');
            }

            // Create courier
            $courierData = [
                'user_id' => $user->id,
                'branch_id' => $data['branch_id'],
                'address' => $data['address'] ?? null,
                'delivery_percent' => $data['delivery_percent'] ?? null,
                'phone' => $data['mobile'] ?? null,
                'name' => $data['name'] ?? null,
                'family' => $data['family'] ?? null,
                'national_code' => $data['national_code'] ?? null,
                'birth_date' => $data['birth_date'] ?? null,
            ];

            $courier = BranchCourier::create($courierData);
            // Add delivery times if provided
            if (!empty($data['delivery_times'])) {
                foreach ($data['delivery_times'] as $time) {
                    DeliveryTime::create([
                        'branch_courier_id' => $courier->id,
                        'start_time'        => $time['start_time'],
                        'end_time'          => $time['end_time'],
                    ]);
                }
            }

            return $courier->load(['user', 'branch', 'deliveryTimes']);
        });
    }
}
