<?php

namespace Modules\Courier\app\Repositories;

use App\Models\Order;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\DB;
use Morilog\Jalali\Jalalian;
use Illuminate\Support\Facades\Log;
class CourierOrderRepository implements CourierOrderRepositoryInterface
{
    protected Order $model;

    public function __construct(Order $model)
    {
        $this->model = $model;
    }

    /**
     * Get orders assigned to a specific courier by user_id.
     */
    public function getCourierOrders(int $userId, array $filters = [], int $perPage = 15): LengthAwarePaginator
    {
        $query = $this->model->where('driver_id', $userId);

        // Apply filters
        $this->applyFilters($query, $filters);

        // Add relationships
        $query->with([
            'user:id,name,family,mobile,email',
            'branch:id,title,address,logo,min_order,vendor_id',
            'driver.user:id,name,family,mobile',
            'deliveryType:id,title',
            'orderCart.variety:id,title,price'
        ]);

        // Order by updated_at desc
        $query->orderBy('updated_at', 'desc');

        // Paginate results
        $orders = $query->paginate($perPage);

        // Add Jalali dates
        $this->addJalaliDates($orders);

        return $orders;
    }

    /**
     * Get order statistics for a courier.
     */
    public function getCourierOrderStats(int $userId, array $filters = []): array
    {
        Log::info('getCourierOrderStats', ['userId' => $userId, 'filters' => $filters]);
        $query = $this->model->where('driver_id', $userId)
            ->where('order_status', '!=', 'wating_payment')
            ->where('order_status', '!=', 'ordering');

        // Apply filters
        $this->applyFilters($query, $filters);

        $stats = [
            'total_orders' => $query->count(),
            'delivered_orders' => (clone $query)->where('order_status', 'delivered')->count(),
            'in_progress_orders' => (clone $query)->whereIn('order_status', ['preparing', 'sending', 'ready_to_send'])->count(),
            'cancelled_orders' => (clone $query)->whereIn('order_status', ['cancelled', 'returned'])->count(),
            'total_earnings' => (clone $query)->where('order_status', 'delivered')->sum('delivery_price'),
            'average_delivery_time' => $this->getAverageDeliveryTime($userId, $filters),
        ];

        return $stats;
    }

    /**
     * Apply filters to the query.
     */
    protected function applyFilters($query, array $filters): void
    {
        if (isset($filters['order_status']) && !empty($filters['order_status'])) {
            $query->where('order_status', $filters['order_status']);
        }

        if (isset($filters['start_date']) && isset($filters['end_date']) &&
            !empty($filters['start_date']) && !empty($filters['end_date'])) {
            $startDate = Jalalian::fromFormat('Y/m/d', $filters['start_date'])->toCarbon();
            $endDate = Jalalian::fromFormat('Y/m/d', $filters['end_date'])->toCarbon();
            $query->where('created_at', '>=', $startDate)->where('created_at', '<=', $endDate);
        }

        if (isset($filters['payment_status']) && !empty($filters['payment_status'])) {
            $query->where('payment_status', $filters['payment_status']);
        }

        if (isset($filters['branch_id']) && !empty($filters['branch_id'])) {
            $query->where('branch_id', $filters['branch_id']);
        }
    }

    /**
     * Add Jalali dates to orders.
     */
    protected function addJalaliDates($orders): void
    {
        foreach ($orders as $order) {
            $order->created_at_jalali = Jalalian::fromCarbon(\Carbon\Carbon::parse($order->created_at))->format('Y/m/d H:i:s');
            $order->updated_at_jalali = Jalalian::fromCarbon(\Carbon\Carbon::parse($order->updated_at))->format('Y/m/d H:i:s');
        }
    }

    /**
     * Get average delivery time for courier.
     */
    protected function getAverageDeliveryTime(int $userId, array $filters): ?float
    {
        $query = $this->model->where('driver_id', $userId)
            ->where('order_status', 'delivered')
            ->whereNotNull('delivery_time');

        $this->applyFilters($query, $filters);

        return $query->avg('delivery_time');
    }
}
