<?php

namespace Modules\Courier\app\Services;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Modules\Courier\app\Repositories\CourierOrderRepositoryInterface;
use Illuminate\Support\Facades\Log;

class CourierOrderService
{
    protected CourierOrderRepositoryInterface $repository;

    public function __construct(CourierOrderRepositoryInterface $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Get orders assigned to a specific courier.
     *
     * @param int $userId
     * @param array $filters
     * @param int $perPage
     * @return LengthAwarePaginator
     * @throws \Exception
     */
    public function getCourierOrders(int $userId, array $filters = [], int $perPage = 15): LengthAwarePaginator
    {
        try {
            return $this->repository->getCourierOrders($userId, $filters, $perPage);
        } catch (\Exception $e) {
            Log::error('خطا در دریافت سفارشات پیک: ' . $e->getMessage(), [
                'user_id' => $userId,
                'filters' => $filters,
                'per_page' => $perPage
            ]);
            throw $e;
        }
    }

    /**
     * Get order statistics for a courier.
     *
     * @param int $userId
     * @param array $filters
     * @return array
     * @throws \Exception
     */
    public function getCourierOrderStats(int $userId, array $filters = []): array
    {
        try {
            return $this->repository->getCourierOrderStats($userId, $filters);
        } catch (\Exception $e) {
            Log::error('خطا در دریافت آمار سفارشات پیک: ' . $e->getMessage(), [
                'user_id' => $userId,
                'filters' => $filters
            ]);
            throw $e;
        }
    }

    /**
     * Validate courier exists and is active.
     *
     * @param int $userId
     * @return bool
     * @throws \Exception
     */
    public function validateCourier(int $userId): bool
    {
        try {
            $courier = \Modules\Courier\app\Models\BranchCourier::where('user_id', $userId)
                ->whereNull('deleted_at')
                ->first();

            if (!$courier) {
                throw new \Exception('پیک یافت نشد یا غیرفعال است');
            }

            return true;
        } catch (\Exception $e) {
            Log::error('خطا در اعتبارسنجی پیک: ' . $e->getMessage(), [
                'user_id' => $userId
            ]);
            throw $e;
        }
    }

    /**
     * Get courier information.
     *
     * @param int $userId
     * @return array
     * @throws \Exception
     */
    public function getCourierInfo(int $userId): array
    {
        try {
            $courier = \Modules\Courier\app\Models\BranchCourier::where('user_id', $userId)
                ->with(['user:id,name,family,mobile', 'branch:id,title'])
                ->first();

            if (!$courier) {
                throw new \Exception('پیک یافت نشد');
            }

            return [
                'id' => $courier->id,
                'name' => $courier->name,
                'family' => $courier->family,
                'mobile' => $courier->phone,
                'delivery_percent' => $courier->delivery_percent,
                'branch' => $courier->branch,
                'user' => $courier->user
            ];
        } catch (\Exception $e) {
            Log::error('خطا در دریافت اطلاعات پیک: ' . $e->getMessage(), [
                'user_id' => $userId
            ]);
            throw $e;
        }
    }
}
