<?php

namespace Modules\Courier\app\Services;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Modules\Courier\app\Models\BranchCourier;
use Modules\Courier\app\Repositories\BranchCourierRepositoryInterface;

class CourierService
{
    protected BranchCourierRepositoryInterface $repository;

    public function __construct(BranchCourierRepositoryInterface $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Get all couriers with filters and pagination.
     *
     * @param array $filters
     * @param int $perPage
     * @return LengthAwarePaginator
     */
    public function getAllCouriers(array $filters = [], int $perPage = 15): LengthAwarePaginator
    {
        return $this->repository->getAll($filters, $perPage);
    }

    /**
     * Find courier by ID.
     *
     * @param int $id
     * @return BranchCourier|null
     */
    public function findCourier(int $id): ?BranchCourier
    {
        return $this->repository->findById($id);
    }

    /**
     * Create a new courier.
     *
     * @param array $data
     * @return BranchCourier
     */
    public function createCourier(array $data): BranchCourier
    {
        return $this->repository->create($data);
    }

    /**
     * Update courier by ID.
     *
     * @param int $id
     * @param array $data
     * @return bool
     */
    public function updateCourier(int $id, array $data): bool
    {
        return $this->repository->update($id, $data);
    }

    /**
     * Delete courier by ID.
     *
     * @param int $id
     * @return bool
     */
    public function deleteCourier(int $id): bool
    {
        return $this->repository->delete($id);
    }

    /**
     * Search for users.
     *
     * @param array $filters
     * @return array
     */
    public function searchUsers(array $filters): array
    {
        return $this->repository->searchUsers($filters);
    }

    /**
     * Search and create courier.
     *
     * @param array $data
     * @return BranchCourier
     */
    public function searchAndCreateCourier(array $data): BranchCourier
    {
        return $this->repository->searchAndCreate($data);
    }
}
