<?php

namespace Modules\User\app\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Address;
use App\Repositories\Address\AddressRepository;
use App\Repositories\Area\AreaRepository;
use Carbon\Carbon;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class AddressController extends Controller
{
    protected $addressRepository;

    public function __construct(AddressRepository $addressRepository)
    {
        $this->addressRepository = $addressRepository;
    }


    public function index(Request $request)
    {
        $addresses = $this->addressRepository->getAddressUser($request->all());
        return response()->json(['data' => $addresses], 200);

    }

    public function show($id)
    {
        $addresses = $this->addressRepository->show($id, ['user_id' => auth()->id()], 'area:id,title,parent_id,slug,lat,lng', 'area.parent:id,title,parent_id,slug,lat,lng');
        return response()->json(['data' => $addresses], 200);

    }


    public function create(Request $request)
    {
        // Validate the request data
        $validator = Validator::make($request->all(), [
            'title' => 'string|max:255',
            'lat' => 'numeric|between:-90,90',
            'lng' => 'numeric|between:-180,180',
            'address' => 'string|max:255',
            'detail' => 'nullable|string',
            'phone_number' => 'nullable|string',
            'area_id' => 'nullable|exists:areas,id',
        ]);

        // If validation fails, return the error response
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Update the user's profile
        $data = $request->all();


        if(!isset($request->area_id)|| $request->area_id==null)
        $data['area_id'] = $this->findNearestArea($request->lat,$request->lng)->id??null;

        $data['user_id']=auth()->id();
        $address = $this->addressRepository->store($data);



        if ($address != null)
            return response()->json(['data' => $address], 200);


        // Return a success response
        return response()->json(['message' => 'مشکلی پیش آمده است لطفا بعدا تلاش فرمایید'], 500);
    }


    public function update($id, Request $request)
    {
        // Validate the request data
        $validator = Validator::make($request->all(), [
            'title' => 'string|max:255',
            'lat' => 'numeric|between:-90,90',
            'lng' => 'numeric|between:-180,180',
            'address' => 'string|max:255',
            'detail' => 'nullable|string',
            'phone_number' => 'nullable|string',
            'area_id' => 'nullable|exists:areas,id',
        ]);

        // If validation fails, return the error response
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Get the authenticated user
        $user = auth()->user();
        $data = $request->all();
        // Update the user's profile
        $data['area_id'] = $this->findNearestArea($request->lat,$request->lng)->id??null;
        $data['user_id']=auth()->id();

        $address = $this->addressRepository->update($id, $data);

        if ($address)
            return response()->json(['data' => $address], 200);


        // Return a success response
        return response()->json(['message' => 'مشکلی پیش آمده است لطفا بعدا تلاش فرمایید'], 500);
    }


    public function destroy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $category = $this->addressRepository->destroy($request->id);
        if (!$category) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 404);
        }
        return response()->json(['message' => 'دسته بندی با موفقیت حذف شد '], 200);
    }
    // function findNearestArea($userLat, $userLon)
    // {
    //     $area = DB::select("
    //         SELECT id, title, lat, lng,
    //             (6371 * acos(cos(radians(?))
    //             * cos(radians(lat))
    //             * cos(radians(lng) - radians(?))
    //             + sin(radians(?))
    //             * sin(radians(lat)))) AS distance
    //         FROM areas
    //         WHERE deleted_at IS NULL
    //         HAVING distance < ?
    //         ORDER BY distance
    //         LIMIT 1
    //     ", [
    //         $userLat,
    //         $userLon,
    //         $userLat,
    //         50 // فاصله حداکثر به کیلومتر (اختیاری)
    //     ]);

    //     return $area ? $area[0] : null;
    // }
    function findNearestArea(float $userLat, float $userLon)
{

    $areas = DB::table('areas')
        ->select('id', 'title', 'lat', 'lng', 'coordinates')
        ->whereNull('deleted_at')
        ->whereNotNull('coordinates')
        ->get();

    foreach ($areas as $area) {
        // coordinates می‌تونه استرینگ JSON یا آرایه باشه
        $coords = is_string($area->coordinates)
            ? json_decode($area->coordinates, true)
            : $area->coordinates;

        if (!is_array($coords) || count($coords) < 3) {
            continue; // چندضلعی معتبر نیست
        }

        if ($this->pointInPolygon($userLat, $userLon, $coords)) {
            // نقطه کاربر داخل این منطقه است: همین را برگردان
            return $area;
        }
    }


    $near = DB::select("
        SELECT id, title, lat, lng,
            (6371 * acos(
                cos(radians(?)) * cos(radians(lat)) *
                cos(radians(lng) - radians(?)) +
                sin(radians(?)) * sin(radians(lat))
            )) AS distance
        FROM areas
        WHERE deleted_at IS NULL
        HAVING distance < ?
        ORDER BY distance
        LIMIT 1
    ", [
        $userLat,
        $userLon,
        $userLat,
        50,
    ]);

    return $near ? $near[0] : null;
}
function pointInPolygon(float $lat, float $lng, array $polygon): bool
    {
        // الگوریتم ray casting. توجه: در داده‌های شما [lat, lng] است.
        // در محاسبه‌ها x=lng و y=lat در نظر می‌گیریم.
        $inside = false;
        $px = $lng;
        $py = $lat;
        $n  = count($polygon);

        if ($n < 3) return false;

        // اگر اولین و آخرین نقطه برابر نبودند، نیازی به بستن حلقه نیست
        // چون از ایندکس‌ها به صورت i/j استفاده می‌کنیم.
        for ($i = 0, $j = $n - 1; $i < $n; $j = $i++) {
            // تبدیل [lat, lng] => (x=lng, y=lat)
            $xi = (float)$polygon[$i][1]; $yi = (float)$polygon[$i][0];
            $xj = (float)$polygon[$j][1]; $yj = (float)$polygon[$j][0];

            $intersect = (($yi > $py) !== ($yj > $py)) &&
                         ($px < (($xj - $xi) * ($py - $yi) / (($yj - $yi) ?: 1e-12) + $xi));

            if ($intersect) $inside = !$inside;
        }

        return $inside;
    }
}
