<?php

namespace App\Console\Commands;

use App\Enums\NotificationType;
use App\Enums\ProductStatus;
use App\Jobs\SendInactiveProductsSms;
use App\Models\Branch;
use App\Models\NotificationMobile;
use App\Models\Product;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SendInactiveProductsReminders extends Command
{
    protected $signature = 'products:send-inactive-reminders {--min-count=5}';
    protected $description = 'ارسال پیامک یادآوری محصولات غیرفعال برای رستوران‌هایی که تعداد محصولات غیرفعالشان بیشتر از حد مشخص است (پیش‌فرض: 5)';

    public function handle(): int
    {
        $minCount = (int)$this->option('min-count');

        $this->info("جستجو برای رستوران‌هایی با حداقل {$minCount} محصول غیرفعال...");

        // دریافت تعداد محصولات غیرفعال برای هر شعبه
        $branchesWithInactiveProducts = Product::select('branch_id', DB::raw('COUNT(*) as inactive_count'))
            ->where('status', ProductStatus::DISABLE)
            ->groupBy('branch_id')
            ->having('inactive_count', '>=', $minCount)
            ->get();

        $this->info("تعداد رستوران‌های یافت شده: " . $branchesWithInactiveProducts->count());

        foreach ($branchesWithInactiveProducts as $branchData) {
            $this->processBranch($branchData->branch_id, $branchData->inactive_count);
        }

        $this->info("عملیات تکمیل شد.");
        return self::SUCCESS;
    }

    private function processBranch($branchId, $inactiveCount): void
    {
        try {
            // دریافت اطلاعات رستوران
            $branch = Branch::find($branchId);

            if (!$branch) {
                Log::warning("رستوران با شناسه #{$branchId} یافت نشد");
                return;
            }

            $restaurantName = $branch->title ?? 'رستوران';

            // دریافت شماره‌های موبایل رستوران برای نوع محصولات غیرفعال
            $restaurantMobiles = NotificationMobile::getMobilesForBranchAndType(
                $branchId,
                NotificationType::PRODUCT_INACTIVE
            );

            // اگر شماره‌ای در notification_mobiles ثبت نشده بود
            if (empty($restaurantMobiles)) {
                // از شماره پیش‌فرض رستوران استفاده می‌کنیم
                $userBranch = \App\Repositories\User\UserRepository::class;
                $branchManager = app($userBranch)->getUserWithBranchId($branchId);

                if ($branchManager && $branchManager->mobile) {
                    $restaurantMobiles = [$branchManager->mobile];
                    $this->line("شماره notification_mobiles یافت نشد، از شماره پیش‌فرض رستوران استفاده شد");
                } else {
                    $this->line("شماره موبایل برای رستوران #{$branchId} ({$restaurantName}) یافت نشد - رد شد");
                    return;
                }
            }

            // ارسال Job برای هر شماره موبایل رستوران
            foreach ($restaurantMobiles as $mobile) {
                SendInactiveProductsSms::dispatch(
                    $inactiveCount,
                    $restaurantName,
                    $mobile
                );
            }

            $this->line("SMS ارسال شد برای رستوران #{$branchId} ({$restaurantName}) - {$inactiveCount} محصول غیرفعال به " . count($restaurantMobiles) . " شماره");

        } catch (\Exception $e) {
            Log::error("خطا در پردازش رستوران #{$branchId}: " . $e->getMessage());
            $this->error("خطا در پردازش رستوران #{$branchId}: " . $e->getMessage());
        }
    }
}

