<?php

namespace App\Console\Commands;

use App\Enums\OrderStatus;
use Illuminate\Console\Command;
use App\Models\Order;
use App\Repositories\SMS\SMSRepositoryInterface;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;

class SendOpenOrderReminders extends Command
{
    protected $signature = 'orders:send-open-reminders {--minutes=30}';
    protected $description = 'Send SMS to the order owner if the order is open for more than N minutes (default: 30).';

    public function handle(): int
    {
        $minutes = (int)$this->option('minutes');
        $cutoff  = Carbon::now()->subMinutes($minutes);

        $totalCount = Order::query()
            ->whereIn('order_status', [
                OrderStatus::ORDERING->value,
                OrderStatus::WATINGPAYMENT->value
            ])
            ->where('updated_at', '<=', $cutoff)
            ->whereNull('open_reminder_sent_at')
            ->count();

        $this->info("جستجو برای سفارشات باز بیش از {$minutes} دقیقه...");
        $this->info("تعداد سفارشات یافت شده: " . $totalCount);

        $sentCount = 0;
        Order::query()
            ->with('user:id,mobile')
            ->whereIn('order_status', [
                OrderStatus::ORDERING->value,
                OrderStatus::WATINGPAYMENT->value
            ])
            ->where('updated_at', '<=', $cutoff)
            ->whereNull('open_reminder_sent_at')
            ->orderBy('id')
            ->chunkById(200, function ($orders) use (&$sentCount) {
                foreach ($orders as $order) {
                    $phone = $order->user->mobile ?? null;
                    if (!$phone) {
                        Log::warning("No phone for order #{$order->id}");
                        continue;
                    }
                    try {
                        app(SMSRepositoryInterface::class)->OpenOrderReminder($order->id, $order->user->mobile);
                        // به‌روزرسانی زمان ارسال یادآوری
                        $order->update(['open_reminder_sent_at' => now()]);
                        $sentCount++;
                    } catch (\Throwable $e) {
                        Log::error("SMS failed for order #{$order->id}: {$e->getMessage()}");
                    }
                }
            });

        $this->info("تعداد پیامک‌های ارسال شده: " . $sentCount);
        return self::SUCCESS;
    }
}
