<?php

namespace App\Console\Commands;

use App\Enums\NotificationType;
use App\Jobs\SendOrderDelayApprovalSms;
use App\Models\NotificationMobile;
use App\Models\Order;
use App\Models\User;
use App\Models\Branch;
use Illuminate\Console\Command;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;

class SendOrderDelayApprovalReminders extends Command
{
    protected $signature = 'orders:send-delay-approval-reminders {--minutes=15}';
    protected $description = 'ارسال پیامک تاخیر تایید سفارش برای رستوران‌هایی که سفارشاتشان بیش از N دقیقه در انتظار تایید است (پیش‌فرض: 15 دقیقه)';

    public function handle(): int
    {
        $minutes = (int)$this->option('minutes');
        $cutoff = Carbon::now()->subMinutes($minutes);

        $this->info("جستجو برای سفارشات تاخیر تایید بیش از {$minutes} دقیقه...");

        $orders = Order::query()
            ->with(['user:id,name,mobile', 'branch:id,title'])
            ->where('order_status', 'awaiting_restaurant_approval')
            ->where('updated_at', '<=', $cutoff)
            ->whereNull('delay_approval_reminder_sent_at')
            ->orderBy('id')
            ->get();

        $this->info("تعداد سفارشات یافت شده: " . $orders->count());

        foreach ($orders as $order) {
            $this->processOrder($order);
        }

        $this->info("عملیات تکمیل شد.");
        return self::SUCCESS;
    }

    private function processOrder(Order $order): void
    {
        try {
            // بررسی وجود کاربر و رستوران
            if (!$order->user || !$order->branch) {
                Log::warning("کاربر یا رستوران برای سفارش #{$order->id} موجود نیست");
                return;
            }

            // دریافت شماره‌های موبایل رستوران برای نوع تاخیر تایید
            $restaurantMobiles = NotificationMobile::getMobilesForBranchAndType(
                $order->branch_id,
                NotificationType::ORDER_DELAY_APPROVAL
            );

            if (empty($restaurantMobiles)) {
                Log::info("شماره موبایل برای رستوران #{$order->branch_id} و نوع ORDER_DELAY_APPROVAL یافت نشد");
                return;
            }

            // آماده کردن اطلاعات برای ارسال SMS
            $customerName = $order->user->name ?? 'مشتری گرامی';
            $orderAmount = number_format($order->order_price) . ' تومان';
            $restaurantName = $order->branch->title ?? 'رستوران';

            // ارسال Job برای هر شماره موبایل رستوران
            foreach ($restaurantMobiles as $mobile) {
                SendOrderDelayApprovalSms::dispatch(
                    $order->id,
                    $customerName,
                    $orderAmount,
                    $restaurantName,
                    $mobile
                );
            }

            // به‌روزرسانی زمان ارسال یادآوری
            $order->update(['delay_approval_reminder_sent_at' => now()]);

            $this->line("SMS ارسال شد برای سفارش #{$order->id} به " . count($restaurantMobiles) . " شماره رستوران");

        } catch (\Exception $e) {
            Log::error("خطا در پردازش سفارش #{$order->id}: " . $e->getMessage());
            $this->error("خطا در پردازش سفارش #{$order->id}: " . $e->getMessage());
        }
    }
}
