<?php

namespace App\Console\Commands;

use App\Enums\NotificationType;
use App\Jobs\SendOrderDelayDeliverySms;
use App\Models\NotificationMobile;
use App\Models\Order;
use App\Models\User;
use Illuminate\Console\Command;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;

class SendOrderDelayDeliveryReminders extends Command
{
    protected $signature = 'orders:send-delay-delivery-reminders {--minutes=30}';
    protected $description = 'ارسال پیامک تاخیر تحویل سفارش برای رستوران‌هایی که سفارشاتشان بیش از N دقیقه در انتظار تحویل است (پیش‌فرض: 30 دقیقه)';

    public function handle(): int
    {
        $minutes = (int)$this->option('minutes');
        $cutoff = Carbon::now()->subMinutes($minutes);

        $this->info("جستجو برای سفارشات تاخیر تحویل بیش از {$minutes} دقیقه...");

        $orders = Order::query()
            ->with(['user:id,name,mobile', 'branch:id,title'])
            ->whereIn('order_status', ['preparing', 'sending', 'ready_to_send'])
            ->where('updated_at', '<=', $cutoff)
            ->whereNull('delay_delivery_reminder_sent_at')
            ->orderBy('id')
            ->get();

        $this->info("تعداد سفارشات یافت شده: " . $orders->count());

        foreach ($orders as $order) {
            $this->processOrder($order);
        }

        $this->info("عملیات تکمیل شد.");
        return self::SUCCESS;
    }

    private function processOrder(Order $order): void
    {
        try {
            // بررسی وجود کاربر و رستوران
            if (!$order->user || !$order->branch) {
                Log::warning("کاربر یا رستوران برای سفارش #{$order->id} موجود نیست");
                return;
            }

            // دریافت شماره‌های موبایل رستوران برای نوع تاخیر تحویل
            $restaurantMobiles = NotificationMobile::getMobilesForBranchAndType(
                $order->branch_id,
                NotificationType::ORDER_DELAY_DELIVERY
            );

            if (empty($restaurantMobiles)) {
                Log::info("شماره موبایل برای رستوران #{$order->branch_id} و نوع ORDER_DELAY_DELIVERY یافت نشد");
                return;
            }

            // آماده کردن اطلاعات برای ارسال SMS
            $customerName = $order->user->name ?? 'مشتری گرامی';

            // ارسال Job برای هر شماره موبایل رستوران
            foreach ($restaurantMobiles as $mobile) {
                SendOrderDelayDeliverySms::dispatch(
                    $order->id,
                    $customerName,
                    $mobile
                );
            }

            // به‌روزرسانی زمان ارسال یادآوری
            $order->update(['delay_delivery_reminder_sent_at' => now()]);

            $this->line("SMS ارسال شد برای سفارش #{$order->id} به " . count($restaurantMobiles) . " شماره رستوران");

        } catch (\Exception $e) {
            Log::error("خطا در پردازش سفارش #{$order->id}: " . $e->getMessage());
            $this->error("خطا در پردازش سفارش #{$order->id}: " . $e->getMessage());
        }
    }
}
