<?php

namespace App\Http\Controllers\V1\Admin;

use App\Http\Controllers\Controller;
use App\Models\Branch;
use App\Repositories\Area\AreaRepository;
use App\Repositories\Area\MaterialRepository;
use App\Repositories\AreaManagment\AreaManagmentRepository;
use App\Repositories\Branch\BranchRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;

class AreaController extends Controller
{
    protected $areaRepository;
    protected $areaManagmentRepository;
    public function __construct()
    {
        $this->areaRepository = app(AreaRepository::class);
        $this->areaManagmentRepository = app(AreaManagmentRepository::class);
    }

    public function index(Request $request)
    {
        $areas = $this->areaRepository->getWithFilter($request->all());
        return response()->json(['data' => $areas], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'slug' => 'nullable|unique:areas',
            'parent_id' => 'nullable|numeric',
            'lat' => 'nullable|numeric|between:-90,90',
            'lng' => 'nullable|numeric|between:-180,180',
            'coordinates' => 'nullable',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        $area = $this->areaRepository->store($request->all());
        return response()->json(['data' => $area], 201);
    }

    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'area_id' => 'required|exists:areas,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }
        $branch = $this->areaRepository->show($request->area_id);

        if (!$branch) {
            return response()->json(['message' => 'منطقه ای یافت نشد'], 404);
        }
        return response()->json(['data' => $branch]);
    }

    public function update(Request $request)
    {
        Log::info('Area update request received');
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'slug' => 'nullable|' . Rule::unique('areas')->ignore($request->area_id),
            'parent_id' => 'nullable|numeric|exists:areas,id',
            'lat' => 'nullable|numeric|between:-90,90',
            'lng' => 'nullable|numeric|between:-180,180',
            'area_id' => 'required|numeric|exists:areas,id',
            'coordinates' => 'nullable',
        ]);
        Log::info($validator->errors());

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        $branch = $this->areaRepository->update($request->area_id, $request->all());
        if ($branch) {
            return response()->json(['message' => 'عملیات با موفقیت انجام شد'], 200);
        }
        return response()->json(['message' => 'عملیات با شکست مواجه شد'], 400);
    }

    public function destroy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'area_id' => 'required|exists:areas,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }
        $branch = $this->areaRepository->destroy($request->area_id);
        if (!$branch) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 404);
        }
        return response()->json(['message' => 'منطقه با موفقیت حذف شد '], 200);
    }

    public function createManagement(Request $request)
    {
        $areaId = $request->input('area_id');
        if ($areaId == 0) {
            $validator = Validator::make($request->all(), [
                'area_id' => 'required',
                'branch_id' => 'required|exists:branches,id',
                'schedules' => 'required|array',
                'schedules.*.day' => 'required|array|min:1',
                'schedules.*.day.*' => 'required|string',
                'schedules.*.service_time_from' => 'required|string',
                'schedules.*.service_time_to' => 'required|string',
                'schedules.*.min_order' => 'required|numeric',
                'schedules.*.status' => 'required|boolean',
                'schedules.*.shipping_methods' => 'required|array',
                'schedules.*.shipping_methods.*' => 'required|exists:delivery_types,id',
            ], [
                'schedules.*.service_time_from.regex' => 'فرمت ساعت شروع نامعتبر است. مثال صحیح: 08:30',
                'schedules.*.service_time_to.regex' => 'فرمت ساعت پایان نامعتبر است. مثال صحیح: 17:30',
                'schedules.*.day.required' => 'انتخاب روز الزامی است',
                'schedules.*.day.min' => 'حداقل یک روز باید انتخاب شود',
                'schedules.*.min_order.required' => 'حداقل سفارش الزامی است',
                'schedules.*.min_order.numeric' => 'حداقل سفارش باید عدد باشد',
                'schedules.*.shipping_methods.required' => 'انتخاب روش ارسال الزامی است',
            ]);
        } else {
            // مدیریت مناطق: ولیدیشن کامل
            $validator = Validator::make($request->all(), [
                'area_id' => [
                    'required',
                    function($attribute, $value, $fail) {
                        if ($value != 0 && !\DB::table('areas')->where('id', $value)->exists()) {
                            $fail('area_id نامعتبر است.');
                        }
                    }
                ],
                'branch_id' => 'required|exists:branches,id',
                'schedules' => 'required|array',
                'schedules.*.day' => 'required|array|min:1',
                'schedules.*.day.*' => 'required|string',
                'schedules.*.service_time_from' => 'required',
                'schedules.*.service_time_to' => 'required',
                'schedules.*.min_order' => 'required|numeric',
                'schedules.*.delivery_time' => 'required|numeric',
                'schedules.*.order_limit_per_half_hour' => 'required|numeric',
                'schedules.*.free_delivery' => 'required|boolean',
                'schedules.*.free_delivery_min_amount' => 'nullable|numeric',
                'schedules.*.status' => 'required|boolean',
                'schedules.*.out_of_range_price' => 'required|numeric',
                'schedules.*.order_amounts' => 'required|array',
                'schedules.*.order_amounts.*.orderAmountDelivery' => 'required|numeric',
                'schedules.*.shipping_methods' => 'required|array',
                'schedules.*.shipping_methods.*' => 'required|exists:delivery_types,id',
            ], [
                'schedules.*.service_time_from.regex' => 'فرمت ساعت شروع نامعتبر است. مثال صحیح: 08:30',
                'schedules.*.service_time_to.regex' => 'فرمت ساعت پایان نامعتبر است. مثال صحیح: 17:30',
                'schedules.*.day.required' => 'انتخاب روز الزامی است',
                'schedules.*.day.min' => 'حداقل یک روز باید انتخاب شود',
                'schedules.*.min_order.required' => 'حداقل سفارش الزامی است',
                'schedules.*.min_order.numeric' => 'حداقل سفارش باید عدد باشد',
                'schedules.*.delivery_time.required' => 'زمان ارسال الزامی است',
                'schedules.*.delivery_time.numeric' => 'زمان ارسال باید عدد باشد',
                'schedules.*.order_limit_per_half_hour.required' => 'محدودیت سفارش الزامی است',
                'schedules.*.order_limit_per_half_hour.numeric' => 'محدودیت سفارش باید عدد باشد',
                'schedules.*.out_of_range_price.required' => 'هزینه خارج از محدوده الزامی است',
                'schedules.*.out_of_range_price.numeric' => 'هزینه خارج از محدوده باید عدد باشد',
                'schedules.*.shipping_methods.required' => 'انتخاب روش ارسال الزامی است',
            ]);
        }


        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        // Validate time ranges
        $timeValidationErrors = $this->validateTimeRanges($request->input('schedules', []));
        if (!empty($timeValidationErrors)) {
            return response()->json(['error' => ['time_ranges' => $timeValidationErrors]], 400);
        }

        try {
            DB::beginTransaction();

            $area = $this->areaManagmentRepository->createManagement($request->all());

            DB::commit();
            return response()->json(['data' => $area, 'message' => 'Area management created successfully'], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating area management: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create area management'], 500);
        }
    }

    /**
     * Validate time ranges for schedules
     */
    private function validateTimeRanges($schedules)
    {
        $errors = [];

        // Validate individual time ranges (from time should not be greater than to time)
        foreach ($schedules as $scheduleIndex => $schedule) {
            if (!empty($schedule['service_time_from']) && !empty($schedule['service_time_to'])) {
                $fromTime = strtotime($schedule['service_time_from']);
                $toTime = strtotime($schedule['service_time_to']);

                if ($fromTime >= $toTime) {
                    $errors[] = "بازه شماره " . ($scheduleIndex + 1) . ": ساعت شروع نمی‌تواند بزرگتر یا مساوی ساعت پایان باشد. (از {$schedule['service_time_from']} تا {$schedule['service_time_to']})";
                }
            }
        }

        // Validate overlapping time ranges for same days
        $dayGroups = [];
        foreach ($schedules as $scheduleIndex => $schedule) {
            if (!empty($schedule['day']) && is_array($schedule['day'])) {
                foreach ($schedule['day'] as $day) {
                    if (!isset($dayGroups[$day])) {
                        $dayGroups[$day] = [];
                    }
                    $dayGroups[$day][] = [
                        'index' => $scheduleIndex,
                        'from' => $schedule['service_time_from'],
                        'to' => $schedule['service_time_to']
                    ];
                }
            }
        }

        foreach ($dayGroups as $day => $ranges) {
            if (count($ranges) > 1) {
                usort($ranges, function($a, $b) {
                    return strtotime($a['from']) - strtotime($b['from']);
                });

                for ($i = 0; $i < count($ranges) - 1; $i++) {
                    $current = $ranges[$i];
                    $next = $ranges[$i + 1];

                    if (!empty($current['from']) && !empty($current['to']) &&
                        !empty($next['from']) && !empty($next['to'])) {

                        $currentEnd = strtotime($current['to']);
                        $nextStart = strtotime($next['from']);

                        if ($currentEnd > $nextStart) {
                            $daysFa = [
                                'Saturday' => 'شنبه',
                                'Sunday' => 'یکشنبه',
                                'Monday' => 'دوشنبه',
                                'Tuesday' => 'سه‌شنبه',
                                'Wednesday' => 'چهارشنبه',
                                'Thursday' => 'پنجشنبه',
                                'Friday' => 'جمعه',
                            ];
                            $dayFa = $daysFa[$day] ?? $day;
                            $errors[] = "تداخل زمانی در روز {$dayFa}: بازه {$current['from']} تا {$current['to']} با بازه {$next['from']} تا {$next['to']} تداخل دارد.";
                        }
                    }
                }
            }
        }

        return $errors;
    }

    public function updateManagement(Request $request)
    {
        Log::info('Area management update request received');
        $areaId = $request->input('area_id');
        if ($areaId == 0) {
            // مدیریت کلی: فقط فیلدهای ضروری
            $validator = Validator::make($request->all(), [
                'area_id' => 'required',
                'branch_id' => 'required|exists:branches,id',
                'schedules' => 'required|array',
                'schedules.*.day' => 'required|array|min:1',
                'schedules.*.day.*' => 'required|string',
                'schedules.*.service_time_from' => 'required|string',
                'schedules.*.service_time_to' => 'required|string',
                'schedules.*.min_order' => 'required|numeric',
                'schedules.*.status' => 'required|boolean',
                'schedules.*.shipping_methods' => 'required|array',
                'schedules.*.shipping_methods.*' => 'required|exists:delivery_types,id',
            ], [
                'schedules.*.service_time_from.regex' => 'فرمت ساعت شروع نامعتبر است. مثال صحیح: 08:30',
                'schedules.*.service_time_to.regex' => 'فرمت ساعت پایان نامعتبر است. مثال صحیح: 17:30',
                'schedules.*.day.required' => 'انتخاب روز الزامی است',
                'schedules.*.day.min' => 'حداقل یک روز باید انتخاب شود',
                'schedules.*.min_order.required' => 'حداقل سفارش الزامی است',
                'schedules.*.min_order.numeric' => 'حداقل سفارش باید عدد باشد',
                'schedules.*.shipping_methods.required' => 'انتخاب روش ارسال الزامی است',
            ]);
        } else {
            // مدیریت مناطق: ولیدیشن کامل
            $validator = Validator::make($request->all(), [
                'area_id' => [
                    'required',
                    function($attribute, $value, $fail) {
                        if ($value != 0 && !DB::table('areas')->where('id', $value)->exists()) {
                            $fail('area_id نامعتبر است.');
                        }
                    }
                ],
                'branch_id' => 'required|exists:branches,id',
                'schedules' => 'required|array',
                'schedules.*.day' => 'required|array|min:1',
                'schedules.*.day.*' => 'required|string',
                'schedules.*.service_time_from' => 'required|string',
                'schedules.*.service_time_to' => 'required|string',
                'schedules.*.min_order' => 'required|numeric',
                'schedules.*.delivery_time' => 'required|numeric',
                'schedules.*.order_limit_per_half_hour' => 'required|numeric',
                'schedules.*.free_delivery' => 'required|boolean',
                'schedules.*.free_delivery_min_amount' => 'nullable|numeric',
                'schedules.*.status' => 'required|boolean',
                'schedules.*.out_of_range_price' => 'required|numeric',
                'schedules.*.order_amounts' => 'required|array',
                'schedules.*.order_amounts.*.orderAmountDelivery' => 'required|numeric',
                'schedules.*.shipping_methods' => 'required|array',
                'schedules.*.shipping_methods.*' => 'required',
            ], [
                'schedules.*.service_time_from.regex' => 'فرمت ساعت شروع نامعتبر است. مثال صحیح: 08:30',
                'schedules.*.service_time_to.regex' => 'فرمت ساعت پایان نامعتبر است. مثال صحیح: 17:30',
                'schedules.*.day.required' => 'انتخاب روز الزامی است',
                'schedules.*.day.min' => 'حداقل یک روز باید انتخاب شود',
                'schedules.*.min_order.required' => 'حداقل سفارش الزامی است',
                'schedules.*.min_order.numeric' => 'حداقل سفارش باید عدد باشد',
                'schedules.*.delivery_time.required' => 'زمان ارسال الزامی است',
                'schedules.*.delivery_time.numeric' => 'زمان ارسال باید عدد باشد',
                'schedules.*.order_limit_per_half_hour.required' => 'محدودیت سفارش الزامی است',
                'schedules.*.order_limit_per_half_hour.numeric' => 'محدودیت سفارش باید عدد باشد',
                'schedules.*.out_of_range_price.required' => 'هزینه خارج از محدوده الزامی است',
                'schedules.*.out_of_range_price.numeric' => 'هزینه خارج از محدوده باید عدد باشد',
                'schedules.*.shipping_methods.required' => 'انتخاب روش ارسال الزامی است',
            ]);
        }
        Log::info('Area management validation passed');
        Log::info($validator->errors());
        if ($validator->fails()) {
            Log::info($validator->errors()->first());

            return response()->json(['error' => $validator->errors()], 400);
        }
        Log::info('Area management update request received');
        // Validate time ranges
        $timeValidationErrors = $this->validateTimeRanges($request->input('schedules', []));
        if (!empty($timeValidationErrors)) {
            return response()->json(['error' => ['time_ranges' => $timeValidationErrors]], 400);
        }

        try {
            DB::beginTransaction();
            Log::info('Area management update request received1');
            $area = $this->areaManagmentRepository->updateManagement($request->all());
            Log::info('Area management update request received2');
            DB::commit();
            return response()->json(['message' => 'Area management updated successfully'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating area management: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update area management'], 500);
        }
    }

    public function getBranchAreas(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        try {
            $areas = $this->areaManagmentRepository->getBranchAreas($request->branch_id);
            return response()->json(['data' => $areas], 200);
        } catch (\Exception $e) {
            Log::error('Error getting branch areas: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to get branch areas'], 500);
        }
    }

    public function showManagement(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
            'area_id' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        try {
            $areaManagement = $this->areaManagmentRepository->showManagement($request->all());

            if (!$areaManagement) {
                return response()->json(['error' => 'Area management not found'], 404);
            }

            return response()->json(['data' => $areaManagement], 200);
        } catch (\Exception $e) {
            Log::error('Error getting area management: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to get area management'], 500);
        }
    }

    public function deleteManagement(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
            'area_id' => 'required|exists:areas,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        try {
            DB::beginTransaction();

            $result = $this->areaManagmentRepository->deleteManagement($request->all());

            if (!$result) {
                DB::rollBack();
                return response()->json(['error' => 'Area management not found'], 404);
            }

            DB::commit();
            return response()->json(['message' => 'Area management deleted successfully'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting area management: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete area management'], 500);
        }
    }

    public function toggleAreaStatus(Request $request)
    {
        Log::info('111');
        $validator = Validator::make($request->all(), [
            'area_id' => 'required',
            'branch_id' => 'required'
        ]);
        Log::info($validator->errors());
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        try {
            $this->areaManagmentRepository->toggleAreaStatus($request->all());
            return response()->json(['message' => 'Area status updated successfully'], 200);
        } catch (\Exception $e) {
            Log::error('Error toggling area status: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update area status'], 500);
        }
    }

    public function getPriceByAddress(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'address_id' => 'required',
            'branch_id' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        $data = $this->areaManagmentRepository->getPriceByAddress($request->all());
        if (!$data) {
            return response()->json(['error' => 'در این زمان امکان ارسال وجود ندارد'], 404);
        }


        return response()->json(['data' => $data], 200);
    }

    public function getAreaBranches(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'area_id' => 'required|array',
            'area_id.*' => 'required|numeric'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }
        try {
            $branches = $this->areaManagmentRepository->getAreaBranches($request->area_id);
            return response()->json(['data' => $branches], 200);
        } catch (\Exception $e) {
            Log::error('Error getting area branches: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to get area branches'], 500);
        }
    }
}
