<?php

namespace App\Http\Controllers\V1\Admin;

use App\Http\Controllers\BaseController;
use App\Http\Requests\AreaSpecialConditionRequest;
use App\Repositories\AreaSpecialCondition\RepositoryInterface;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class AreaSpecialConditionController extends BaseController
{
    protected $areaSpecialConditionRepository;

    public function __construct()
    {
        $this->areaSpecialConditionRepository = app(RepositoryInterface::class);
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'branch_id' => 'nullable|numeric|exists:branches,id',
                'area_id' => 'nullable|numeric|exists:areas,id',
                'date' => 'nullable|date',
                'start_date' => 'nullable|date',
                'end_date' => 'nullable|date|after_or_equal:start_date',
                'status' => 'nullable|string|in:active,inactive',
                'search' => 'nullable|string|max:255',
                'per_page' => 'nullable|integer|min:1|max:100',
                'sort_column' => 'nullable|string',
                'sort_type' => 'nullable|string|in:asc,desc',
            ]);

            if ($validator->fails()) {
                return $this->sendError('خطا در اعتبارسنجی', $validator->errors(), 400);
            }

            $conditions = $this->areaSpecialConditionRepository->getWithFilter($request->all(), true);

            // Add branches and areas data to each condition
            if (isset($conditions->data) && is_array($conditions->data)) {
                foreach ($conditions->data as $condition) {
                    $condition->branches_data = $condition->branches();
                    $condition->areas_data = $condition->areas();
                }
            } elseif (is_object($conditions) && property_exists($conditions, 'data') && is_array($conditions->data)) {
                foreach ($conditions->data as $condition) {
                    $condition->branches_data = $condition->branches();
                    $condition->areas_data = $condition->areas();
                }
            } elseif (is_iterable($conditions)) {
                foreach ($conditions as $condition) {
                    if (is_object($condition) && method_exists($condition, 'branches')) {
                        $condition->branches_data = $condition->branches();
                        $condition->areas_data = $condition->areas();
                    }
                }
            }

            return $this->sendResponse($conditions, 'لیست شرایط ویژه با موفقیت دریافت شد');
        } catch (\Exception $e) {
            Log::error('Error in AreaSpecialConditionController@index: ' . $e->getMessage());
            return $this->sendError('خطا در دریافت لیست شرایط ویژه', [], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(AreaSpecialConditionRequest $request)
    {
        try {
            Log::info('Request data: ', $request->all());

            $data = $request->validated();
            Log::info('Validated data: ', $data);

            $data['status'] = $data['status'] ?? 'active';

            // Convert branch_ids and area_ids arrays to the correct field names
            $data['branch_id'] = $data['branch_ids'];
            $data['area_id'] = $data['area_ids'];
            unset($data['branch_ids'], $data['area_ids']);

            $condition = $this->areaSpecialConditionRepository->store($data);

            if ($condition) {
                // Load the related branches and areas
                $condition->branches_data = $condition->branches();
                $condition->areas_data = $condition->areas();

                return $this->sendResponse($condition, 'شرایط ویژه با موفقیت ایجاد شد', true, 201);
            }

            return $this->sendError('خطا در ایجاد شرایط ویژه', [], 500);
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Validation error in AreaSpecialConditionController@store: ', $e->errors());
            return $this->sendError('خطا در اعتبارسنجی داده‌ها', $e->errors(), 422);
        } catch (\Exception $e) {
            Log::error('Error in AreaSpecialConditionController@store: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return $this->sendError('خطا در ایجاد شرایط ویژه', ['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request)
    {
        try {
            Log::info('Show request data: ', $request->all());

            $validator = Validator::make($request->all(), [
                'id' => 'required|numeric|exists:area_special_conditions,id'
            ]);

            if ($validator->fails()) {
                return $this->sendError('خطا در اعتبارسنجی', $validator->errors(), 400);
            }

            $condition = $this->areaSpecialConditionRepository->show($request->id);

            if (!$condition) {
                return $this->sendError('شرایط ویژه یافت نشد', [], 404);
            }

            // Add related branches and areas data
            $condition->branches_data = $condition->branches();
            $condition->areas_data = $condition->areas();

            return $this->sendResponse($condition, 'شرایط ویژه با موفقیت دریافت شد');
        } catch (\Exception $e) {
            Log::error('Error in AreaSpecialConditionController@show: ' . $e->getMessage());
            return $this->sendError('خطا در دریافت شرایط ویژه', [], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(AreaSpecialConditionRequest $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'id' => 'required|numeric|exists:area_special_conditions,id'
            ]);
            if ($validator->fails()) {
                return $this->sendError('خطا در اعتبارسنجی', $validator->errors(), 400);
            }

            Log::info('Update request data: ', $request->all());

            $data = $request->validated();
            Log::info('Update validated data: ', $data);

            $conditionId = $request->id;

            // Convert branch_ids and area_ids arrays to the correct field names
            $data['branch_id'] = $data['branch_ids'];
            $data['area_id'] = $data['area_ids'];
            unset($data['branch_ids'], $data['area_ids']);

            $result = $this->areaSpecialConditionRepository->update($conditionId, $data);

            if ($result) {
                $condition = $this->areaSpecialConditionRepository->show($conditionId);
                // Load the related branches and areas
                $condition->branches_data = $condition->branches();
                $condition->areas_data = $condition->areas();

                return $this->sendResponse($condition, 'شرایط ویژه با موفقیت به‌روزرسانی شد');
            }

            return $this->sendError('خطا در به‌روزرسانی شرایط ویژه', [], 500);
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Validation error in AreaSpecialConditionController@update: ', $e->errors());
            return $this->sendError('خطا در اعتبارسنجی داده‌ها', $e->errors(), 422);
        } catch (\Exception $e) {
            Log::error('Error in AreaSpecialConditionController@update: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return $this->sendError('خطا در به‌روزرسانی شرایط ویژه', ['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'id' => 'required|numeric|exists:area_special_conditions,id'
            ]);

            if ($validator->fails()) {
                return $this->sendError('خطا در اعتبارسنجی', $validator->errors(), 400);
            }

            $result = $this->areaSpecialConditionRepository->destroy($request->id);

            if ($result) {
                return $this->sendResponse([], 'شرایط ویژه با موفقیت حذف شد');
            }

            return $this->sendError('خطا در حذف شرایط ویژه', [], 500);
        } catch (\Exception $e) {
            Log::error('Error in AreaSpecialConditionController@destroy: ' . $e->getMessage());
            return $this->sendError('خطا در حذف شرایط ویژه', [], 500);
        }
    }

    /**
     * Get special conditions for a specific branch
     */
    public function getByBranch(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'branch_id' => 'required|numeric|exists:branches,id',
                'date' => 'nullable|date',
                'status' => 'nullable|string|in:active,inactive',
            ]);

            if ($validator->fails()) {
                return $this->sendError('خطا در اعتبارسنجی', $validator->errors(), 400);
            }

            $conditions = $this->areaSpecialConditionRepository->getByBranch($request->branch_id, $request->all());

            // Add branches and areas data to each condition
            if (is_iterable($conditions)) {
                foreach ($conditions as $condition) {
                    if (is_object($condition) && method_exists($condition, 'branches')) {
                        $condition->branches_data = $condition->branches();
                        $condition->areas_data = $condition->areas();
                    }
                }
            }

            return $this->sendResponse($conditions, 'شرایط ویژه شعبه با موفقیت دریافت شد');
        } catch (\Exception $e) {
            Log::error('Error in AreaSpecialConditionController@getByBranch: ' . $e->getMessage());
            return $this->sendError('خطا در دریافت شرایط ویژه شعبه', [], 500);
        }
    }

    /**
     * Get special conditions for a specific area
     */
    public function getByArea(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'area_id' => 'required|numeric|exists:areas,id',
                'date' => 'nullable|date',
                'status' => 'nullable|string|in:active,inactive',
            ]);

            if ($validator->fails()) {
                return $this->sendError('خطا در اعتبارسنجی', $validator->errors(), 400);
            }

            $conditions = $this->areaSpecialConditionRepository->getByArea($request->area_id, $request->all());

            // Add branches and areas data to each condition
            if (is_iterable($conditions)) {
                foreach ($conditions as $condition) {
                    if (is_object($condition) && method_exists($condition, 'branches')) {
                        $condition->branches_data = $condition->branches();
                        $condition->areas_data = $condition->areas();
                    }
                }
            }

            return $this->sendResponse($conditions, 'شرایط ویژه منطقه با موفقیت دریافت شد');
        } catch (\Exception $e) {
            Log::error('Error in AreaSpecialConditionController@getByArea: ' . $e->getMessage());
            return $this->sendError('خطا در دریافت شرایط ویژه منطقه', [], 500);
        }
    }

    /**
     * Get currently active special conditions
     */
    public function getCurrentlyActive(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'branch_id' => 'nullable|numeric|exists:branches,id',
                'area_id' => 'nullable|numeric|exists:areas,id',
            ]);

            if ($validator->fails()) {
                return $this->sendError('خطا در اعتبارسنجی', $validator->errors(), 400);
            }

            $conditions = $this->areaSpecialConditionRepository->getCurrentlyActive($request->all());

            // Add branches and areas data to each condition
            if (is_iterable($conditions)) {
                foreach ($conditions as $condition) {
                    if (is_object($condition) && method_exists($condition, 'branches')) {
                        $condition->branches_data = $condition->branches();
                        $condition->areas_data = $condition->areas();
                    }
                }
            }

            return $this->sendResponse($conditions, 'شرایط ویژه فعال فعلی با موفقیت دریافت شد');
        } catch (\Exception $e) {
            Log::error('Error in AreaSpecialConditionController@getCurrentlyActive: ' . $e->getMessage());
            return $this->sendError('خطا در دریافت شرایط ویژه فعال فعلی', [], 500);
        }
    }

    /**
     * Toggle status of a special condition
     */
    public function toggleStatus(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'id' => 'required|numeric|exists:area_special_conditions,id'
            ]);

            if ($validator->fails()) {
                return $this->sendError('خطا در اعتبارسنجی', $validator->errors(), 400);
            }

            $condition = $this->areaSpecialConditionRepository->show($request->id);

            if (!$condition) {
                return $this->sendError('شرایط ویژه یافت نشد', [], 404);
            }

            $newStatus = $condition->status === 'active' ? 'inactive' : 'active';
            $result = $this->areaSpecialConditionRepository->update($request->id, ['status' => $newStatus]);

            if ($result) {
                $condition = $this->areaSpecialConditionRepository->show($request->id);
                // Add related branches and areas data
                $condition->branches_data = $condition->branches();
                $condition->areas_data = $condition->areas();
                return $this->sendResponse($condition, "وضعیت شرایط ویژه به {$newStatus} تغییر یافت");
            }

            return $this->sendError('خطا در تغییر وضعیت شرایط ویژه', [], 500);
        } catch (\Exception $e) {
            Log::error('Error in AreaSpecialConditionController@toggleStatus: ' . $e->getMessage());
            return $this->sendError('خطا در تغییر وضعیت شرایط ویژه', [], 500);
        }
    }
}
