<?php

namespace App\Http\Controllers\V1\Admin;

use App\Http\Controllers\Controller;
use App\Repositories\BankAccount\BankAccountRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class BankAccountController extends Controller
{
    protected $bankAccountRepository;

    public function __construct()
    {
        $this->bankAccountRepository = app(BankAccountRepository::class);
    }

    public function index(Request $request)
    {
        $data = $this->bankAccountRepository->getWithFilter($request->all(),true);
        return response()->json(['data' => $data], 200);
    }

    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'bank_account_id' => 'required|numeric|exists:bank_accounts,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $bankAccount = $this->bankAccountRepository->findById($request->bank_account_id);
        if (!$bankAccount) {
            return response()->json(['message' => 'حساب بانکی یافت نشد'], 404);
        }
        return response()->json(['data' => $bankAccount]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|numeric',
            'title' => 'required',
            'bank_name' => 'required',
            'account_holder_name' => 'required',
            'account_number' => 'required',
            'card_number' => 'required',
            'shaba_number' => 'required',
            'is_active' => 'nullable|boolean'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $bankAccount = $this->bankAccountRepository->storeByAdmin($request->all());
        return response()->json(['data' => $bankAccount, 'message' => 'حساب بانکی با موفقیت ایجاد شد'], 201);
    }

    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'bank_account_id' => 'required|numeric|exists:bank_accounts,id',
            'title' => 'required',
            'bank_name' => 'required',
            'account_holder_name' => 'required',
            'account_number' => 'required',
            'card_number' => 'required',
            'shaba_number' => 'required',
            'is_active' => 'nullable|boolean'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $bankAccount = $this->bankAccountRepository->updateByAdmin($request->bank_account_id, $request->all());
        if ($bankAccount) {
            return response()->json(['message' => 'عملیات با موفقیت انجام شد'], 200);
        }
        return response()->json(['message' => 'عملیات با شکست مواجه شد'], 400);
    }

    public function destroy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'bank_account_id' => 'required|numeric|exists:bank_accounts,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $bankAccount = $this->bankAccountRepository->destroyByAdmin($request->bank_account_id);
        if (!$bankAccount) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 404);
        }
        return response()->json(['message' => 'با موفقیت حذف شد'], 200);
    }

    /**
     * Toggle is_active status for bank account
     */
    public function toggleStatus(Request $request)
    {
        Log::info('1111111111');
        $validator = Validator::make($request->all(), [
            'bank_account_id' => 'required|numeric|exists:bank_accounts,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $bankAccount = $this->bankAccountRepository->toggleStatus($request->bank_account_id);

        if (!$bankAccount) {
            return response()->json(['message' => 'حساب بانکی یافت نشد'], 404);
        }

        $status = $bankAccount->is_active ? 'فعال' : 'غیرفعال';
        return response()->json([
            'data' => $bankAccount,
            'message' => "وضعیت حساب بانکی با موفقیت به {$status} تغییر یافت"
        ], 200);
    }
}
