<?php

namespace App\Http\Controllers\V1\Admin;

use App\Http\Controllers\Controller;
use App\Repositories\User\UserRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class CashBackController extends Controller
{
    protected $userRepository;

    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
    }

    /**
     * User Cash Back Balances
     */
    public function getUserCashBackBalance(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|integer|exists:users,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 400,
                    'message' => 'خطا در اعتبارسنجی داده‌ها',
                    'errors' => $validator->errors()
                ], 400);
            }

            $userId = $request->user_id;
            $cashBackBalances = $this->userRepository->getUserCashBackBalances($userId);
            return response()->json([
                'status' => 200,
                'message' => 'موجودی cash back کاربر با موفقیت دریافت شد',
                'data' => [
                    'user_id' => $userId,
                    'balances' => $cashBackBalances
                ]
            ], 200);
        } catch (\Exception $e) {

            return response()->json([
                'status' => 500,
                'message' => 'خطا در دریافت موجودی cash back',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * User Cash Back Balance by Branch
     */
    public function getUserCashBackBalanceByBranch(Request $request, $branchId)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|integer|exists:users,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 400,
                    'message' => 'خطا در اعتبارسنجی داده‌ها',
                    'errors' => $validator->errors()
                ], 400);
            }

            $userId = $request->user_id;
            $balance = $this->userRepository->getWalletUserInBranch($branchId, $userId);

            return response()->json([
                'status' => 200,
                'message' => 'موجودی cash back کاربر با موفقیت دریافت شد',
                'data' => [
                    'branch_id' => $branchId,
                    'user_id' => $userId,
                    'balance' => $balance,
                    'has_balance' => $balance > 0
                ]
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'خطا در دریافت موجودی cash back',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * User Cash Back Details
     */
    public function getUserCashBackDetails(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|integer|exists:users,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 400,
                    'message' => 'خطا در اعتبارسنجی داده‌ها',
                    'errors' => $validator->errors()
                ], 400);
            }

            $userId = $request->user_id;
            $cashBackDetails = $this->userRepository->getUserCashBackDetails($userId);

            return response()->json([
                'status' => 200,
                'message' => 'جزئیات cash back کاربر با موفقیت دریافت شد',
                'data' => [
                    'user_id' => $userId,
                    'details' => $cashBackDetails
                ]
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'خطا در دریافت جزئیات cash back',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * All Users Cash Back
     */
    public function getAllUsersCashBack(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'page' => 'nullable|integer|min:1',
                'per_page' => 'nullable|integer|min:1|max:100',
                'search' => 'nullable|string|max:255',
                'branch_id' => 'nullable|integer|exists:branches,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 400,
                    'message' => 'خطا در اعتبارسنجی داده‌ها',
                    'errors' => $validator->errors()
                ], 400);
            }

            $usersCashBack = $this->userRepository->getAllUsersCashBack(
                $request->page ?? 1,
                $request->per_page ?? 20,
                $request->search,
                $request->branch_id
            );

            return response()->json([
                'status' => 200,
                'message' => 'لیست کاربران با موجودی cash back با موفقیت دریافت شد',
                'data' => $usersCashBack
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'خطا در دریافت لیست کاربران',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Cash Back Statistics
     */
    public function getCashBackStatistics()
    {
        try {
            $statistics = $this->userRepository->getCashBackStatistics();

            return response()->json([
                'status' => 200,
                'message' => 'آمار cash back با موفقیت دریافت شد',
                'data' => $statistics
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'خطا در دریافت آمار cash back',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
