<?php

namespace App\Http\Controllers\V1\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Repositories\Category\CategoryRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class CategoryController extends Controller
{
    protected $categoryRepository;

    public function __construct()
    {
        $this->categoryRepository = app(CategoryRepository::class);
    }

    public function index(Request $request)
    {
        $categories = $this->categoryRepository->getWithFilter($request->all());
        return response()->json(['data' => $categories], 200);
    }

    public function getWithProducts(Request $request)
    {
        $categories = $this->categoryRepository->getCategoryAllWithProduct($request->all());
        return response()->json(['data' => $categories], 200);
    }


    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'img' => 'nullable|url',
            'start_serving' => 'nullable|date_format:H:i:s',
            'end_serving' => 'nullable|date_format:H:i:s|after_or_equal:start_serving',
            'parent_id' => 'nullable',
            'branch_id' => 'nullable',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $category = $this->categoryRepository->store($request->all());
        return response()->json(['data' => $category], 201);
    }

    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $category = $this->categoryRepository->show($request->id);
        if (!$category) {
            return response()->json(['message' => 'دسته بندی یافت نشد'], 404);
        }
        return response()->json(['data' => $category]);
    }

    public function update(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'img' => 'required|url',
            'start_serving' => 'nullable|date_format:H:i:s',
            'end_serving' => 'nullable|date_format:H:i:s|after_or_equal:start_serving',
            'parent_id' => 'nullable',
            'category_id' => 'required|numeric|exists:categories,id',

        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $category = $this->categoryRepository->update($request->category_id, $request->all());
        if ($category) {
            return response()->json(['message' => 'عملیات با موفقیت انجام شد'], 200);
        }
        return response()->json(['message' => 'عملیات با شکست مواجه شد'], 500);
    }

    public function destroy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $category = $this->categoryRepository->destroy($request->id);
        if (!$category) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 404);
        }
        return response()->json(['message' => 'دسته بندی با موفقیت حذف شد '], 200);
    }

    public function toggleStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|numeric|exists:categories,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $category = $this->categoryRepository->toggleStatus($request->id);

        if (!$category) {
            return response()->json(['message' => 'دسته بندی یافت نشد'], 404);
        }

        $statusText = $category->status === 'active' ? 'فعال' : 'غیرفعال';

        return response()->json([
            'message' => "وضعیت دسته بندی با موفقیت به {$statusText} تغییر یافت",
            'data' => [
                'id' => $category->id,
                'title' => $category->title,
                'status' => $category->status
            ]
        ], 200);
    }

}
