<?php

namespace App\Http\Controllers\V1\Admin;

use App\Enums\CouponType;
use App\Http\Controllers\Controller;
use App\Repositories\Coupon\CouponRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Enum;

class CouponController extends Controller
{
    protected $couponRepository;

    public function __construct()
    {
        $this->couponRepository = app(CouponRepository::class);
    }

    public function index(Request $request)
    {
        $coupons = $this->couponRepository->getWithFilter($request->all(), true);
        return response()->json(['data' => $coupons]);
    }

  

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|max:255',
            'code' => 'required|unique:coupons,code',
            'coupon_id' => 'nullable',
            'type' => 'nullable|integer|between:0,1',
            'percent' => 'nullable|integer|min:1|max:100',
            'price' => 'required_if:percent,==,null|numeric|min:1000',
            'active_from' => 'nullable',
            'active_to' => 'nullable',
            'limit_uses' => 'nullable',
            'limit_user' => 'required|boolean',
            'limit_coupon' => 'nullable|boolean',
            'coupon_ids' => 'required_if:limit_coupon,==,true|array',
            'coupon_ids.*' => 'nullable|exists:coupons,id',
            'user_ids' => 'required_if:limit_user,==,true|array',
            'user_ids.*' => 'nullable|exists:users,id',
            'used_count' => 'nullable|integer|min:1',
            'max_price_discount' => 'nullable',
            'min_price_order' => 'nullable|numeric|min:1000',
            'percent_system' => 'required|numeric|max:100',
            'percent_branch' => 'required|numeric|max:100',
            'condition_type' => [new Enum(CouponType::class)],
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $data = $request->all();
        if (isset($data['active_from']) and $data['active_from'] !== null) {
            $data['active_from'] = \Morilog\Jalali\Jalalian::fromFormat('Y/m/d', $data['active_from'])->toCarbon();
        }

        if (isset($data['active_to']) and $data['active_to'] !== null) {
            $data['active_to'] = \Morilog\Jalali\Jalalian::fromFormat('Y/m/d', $data['active_to'])->toCarbon();
        }
        $coupon = $this->couponRepository->storeA($data);
        if ($coupon) {
            return response()->json(['data' => $coupon], 201);
        }
        return response()->json(['message' => 'عملیات با شکست مواجه شد'], 400);
    }


    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'coupon_id' => 'required|exists:coupons,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $coupon = $this->couponRepository->show($request->coupon_id);
        if (!$coupon) {
            return response()->json(['message' => 'منطقه ای یافت نشد'], 404);
        }
        return response()->json(['data' => $coupon]);
    }

    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|max:255',
            'code' => 'required|unique:coupons,code,'.$request->coupon_id,
            'type' => 'nullable|integer|between:0,1',
            'percent' => 'nullable|integer|min:1|max:100',
            'price' => 'required_if:percent,==,null|numeric|min:1000',
            'active_from' => 'nullable',
            'active_to' => 'nullable',
            'limit_uses' => 'nullable',
            'limit_user' => 'required|boolean',
            'user_ids' => 'required_if:limit_user,==,true|array',
            'user_ids.*' => 'nullable|exists:users,id',
            'used_count' => 'nullable|integer|min:1',
            'max_price_discount' => 'nullable',
            'min_price_order' => 'nullable|numeric|min:1000',
            'percent_system' => 'required|numeric|max:100',
            'percent_branch' => 'required|numeric|max:100',
            'condition_type' => [new Enum(CouponType::class)],
            'coupon_id' => 'required|numeric|exists:coupons,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $data = $request->all();
        if (isset($data['active_from']) and $data['active_from'] !== null) {
            $data['active_from'] = \Morilog\Jalali\Jalalian::fromFormat('Y/m/d', $data['active_from'])->toCarbon();
        }

        if (isset($data['active_to']) and $data['active_to'] !== null) {
            $data['active_to'] = \Morilog\Jalali\Jalalian::fromFormat('Y/m/d', $data['active_to'])->toCarbon();
        }
        $coupon = $this->couponRepository->update($request->coupon_id, $data);
        if ($coupon) {
            return response()->json(['message' => 'عملیات با موفقیت انجام شد'], 200);
        }
        return response()->json(['message' => 'عملیات با شکست مواجه شد'], 400);
    }

    public function destroy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'coupon_id' => 'required|exists:areas,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $coupon = $this->couponRepository->destroy($request->coupon_id);
        if (!$coupon) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 404);
        }
        return response()->json(['message' => 'با موفقیت حذف شد '], 200);
    }


    public function checkCoupon(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'code' => '',
            'user_id' => '',
        ]);
        $coupons = $this->couponRepository->getWithFilter($request->all(), true);
        return response()->json(['data' => $coupons]);
    }


}
