<?php

namespace App\Http\Controllers\V1\Admin;

use App\Enums\NotificationType;
use App\Http\Controllers\Controller;
use App\Repositories\NotificationMobile\NotificationMobileRepositoryInterface;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class NotificationMobileController extends Controller
{
    protected $notificationMobileRepository;

    public function __construct(NotificationMobileRepositoryInterface $notificationMobileRepository)
    {
        $this->notificationMobileRepository = $notificationMobileRepository;
    }

    /**
     * نمایش لیست شماره‌های نوتیفیکیشن
     */
    public function index(Request $request)
    {
        try {
            $filters = $request->only(['branch_id', 'type', 'is_active', 'mobile']);
            $notificationMobiles = $this->notificationMobileRepository->getPaginated($filters);

            return response()->json([
                'data' => $notificationMobiles->items(),
                'pagination' => [
                    'current_page' => $notificationMobiles->currentPage(),
                    'last_page' => $notificationMobiles->lastPage(),
                    'per_page' => $notificationMobiles->perPage(),
                    'total' => $notificationMobiles->total(),
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('NotificationMobile index error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت لیست شماره‌های نوتیفیکیشن'], 500);
        }
    }

    /**
     * نمایش جزئیات یک شماره نوتیفیکیشن
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'notification_mobile_id' => 'required|numeric|exists:notification_mobiles,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            $notificationMobile = $this->notificationMobileRepository->findById($request->notification_mobile_id);

            if (!$notificationMobile) {
                return response()->json(['message' => 'شماره نوتیفیکیشن یافت نشد'], 404);
            }

            return response()->json(['data' => $notificationMobile]);
        } catch (\Exception $e) {
            Log::error('NotificationMobile show error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت جزئیات شماره نوتیفیکیشن'], 500);
        }
    }

    /**
     * ایجاد شماره نوتیفیکیشن جدید
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|numeric|exists:branches,id',
            'type' => 'required|string|in:' . implode(',', NotificationType::getAll()),
            'mobile' => 'required|string|max:20|regex:/^09[0-9]{9}$/',
            'is_active' => 'nullable|boolean'
        ], [
            'branch_id.required' => 'شناسه شعبه الزامی است',
            'branch_id.exists' => 'شعبه یافت نشد',
            'type.required' => 'نوع نوتیفیکیشن الزامی است',
            'type.in' => 'نوع نوتیفیکیشن نامعتبر است',
            'mobile.required' => 'شماره موبایل الزامی است',
            'mobile.regex' => 'فرمت شماره موبایل نامعتبر است',
            'mobile.max' => 'شماره موبایل نباید بیشتر از 20 کاراکتر باشد',
            'is_active.boolean' => 'وضعیت فعال باید true یا false باشد'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            DB::beginTransaction();

            $data = [
                'branch_id' => $request->branch_id,
                'type' => NotificationType::from($request->type),
                'mobile' => $request->mobile,
                'is_active' => $request->is_active ?? true
            ];

            $notificationMobile = $this->notificationMobileRepository->create($data);

            if (!$notificationMobile) {
                DB::rollBack();
                return response()->json(['message' => 'خطا در ایجاد شماره نوتیفیکیشن'], 400);
            }

            DB::commit();

            return response()->json([
                'data' => $notificationMobile->load('branch'),
                'message' => 'شماره نوتیفیکیشن با موفقیت ایجاد شد'
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('NotificationMobile store error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در ایجاد شماره نوتیفیکیشن'], 500);
        }
    }

    /**
     * بروزرسانی شماره نوتیفیکیشن
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'notification_mobile_id' => 'required|numeric|exists:notification_mobiles,id',
            'branch_id' => 'nullable|numeric|exists:branches,id',
            'type' => 'nullable|string|in:' . implode(',', NotificationType::getAll()),
            'mobile' => 'nullable|string|max:20|regex:/^09[0-9]{9}$/',
            'is_active' => 'nullable|boolean'
        ], [
            'notification_mobile_id.required' => 'شناسه شماره نوتیفیکیشن الزامی است',
            'notification_mobile_id.exists' => 'شماره نوتیفیکیشن یافت نشد',
            'branch_id.nullable' => 'شناسه شعبه الزامی است',
            'branch_id.exists' => 'شعبه یافت نشد',
            'type.nullable' => 'نوع نوتیفیکیشن الزامی است',
            'type.in' => 'نوع نوتیفیکیشن نامعتبر است',
            'mobile.nullable' => 'شماره موبایل الزامی است',
            'mobile.regex' => 'فرمت شماره موبایل نامعتبر است',
            'mobile.max' => 'شماره موبایل نباید بیشتر از 20 کاراکتر باشد',
            'is_active.boolean' => 'وضعیت فعال باید true یا false باشد'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            DB::beginTransaction();
            $data = $request->all();
            // $data = [
            //     'branch_id' => $request->branch_id ? $request->branch_id : null,
            //     'type' => $request->type ? NotificationType::from($request->type) : null,
            //     'mobile' => $request->mobile ? $request->mobile : null,
            //     'is_active' => $request->is_active ?? true
            // ];

            $result = $this->notificationMobileRepository->update($request->notification_mobile_id, $data);

            if (!$result) {
                DB::rollBack();
                return response()->json(['message' => 'خطا در بروزرسانی شماره نوتیفیکیشن'], 400);
            }

            DB::commit();

            // دریافت رکورد بروزرسانی شده
            $updatedNotificationMobile = $this->notificationMobileRepository->findById($request->notification_mobile_id);

            return response()->json([
                'data' => $updatedNotificationMobile,
                'message' => 'شماره نوتیفیکیشن با موفقیت بروزرسانی شد'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('NotificationMobile update error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در بروزرسانی شماره نوتیفیکیشن'], 500);
        }
    }

    /**
     * حذف شماره نوتیفیکیشن
     */
    public function destroy(Request $request)
    {
        Log::info('NotificationMobile destroy request: ' . json_encode($request->all()));
        $validator = Validator::make($request->all(), [
            'notification_mobile_id' => 'required|numeric|exists:notification_mobiles,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            DB::beginTransaction();

            $result = $this->notificationMobileRepository->delete($request->notification_mobile_id);

            if (!$result) {
                DB::rollBack();
                return response()->json(['message' => 'خطا در حذف شماره نوتیفیکیشن'], 400);
            }

            DB::commit();

            return response()->json(['message' => 'شماره نوتیفیکیشن با موفقیت حذف شد']);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('NotificationMobile destroy error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در حذف شماره نوتیفیکیشن'], 500);
        }
    }

    /**
     * دریافت شماره‌های نوتیفیکیشن بر اساس شعبه
     */
    public function getByBranch(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|numeric|exists:branches,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            $notificationMobiles = $this->notificationMobileRepository->getByBranch($request->branch_id);
            return response()->json(['data' => $notificationMobiles]);
        } catch (\Exception $e) {
            Log::error('NotificationMobile getByBranch error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت شماره‌های نوتیفیکیشن شعبه'], 500);
        }
    }

    /**
     * دریافت شماره‌های نوتیفیکیشن بر اساس نوع
     */
    public function getByType(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'type' => 'required|string|in:' . implode(',', NotificationType::getAll())
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            $type = NotificationType::from($request->type);
            $notificationMobiles = $this->notificationMobileRepository->getByType($type);
            return response()->json(['data' => $notificationMobiles]);
        } catch (\Exception $e) {
            Log::error('NotificationMobile getByType error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت شماره‌های نوتیفیکیشن بر اساس نوع'], 500);
        }
    }

    /**
     * دریافت شماره‌های موبایل برای ارسال پیامک
     */
    public function getMobileNumbers(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|numeric|exists:branches,id',
            'type' => 'required|string|in:' . implode(',', NotificationType::getAll()),
            'active_only' => 'nullable|boolean'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            $type = NotificationType::from($request->type);

            if ($request->active_only) {
                $mobileNumbers = $this->notificationMobileRepository->getActiveMobileNumbers($request->branch_id, $type);
            } else {
                $mobileNumbers = $this->notificationMobileRepository->getMobileNumbers($request->branch_id, $type);
            }

            return response()->json([
                'data' => [
                    'mobile_numbers' => $mobileNumbers,
                    'count' => count($mobileNumbers)
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('NotificationMobile getMobileNumbers error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت شماره‌های موبایل'], 500);
        }
    }

    /**
     * دریافت انواع نوتیفیکیشن موجود
     */
    public function getNotificationTypes()
    {
        try {
            $types = [];
            foreach (NotificationType::cases() as $type) {
                $types[] = [
                    'value' => $type->value,
                    'label' => $type->getLabel(),
                    'description' => $type->getDescription()
                ];
            }

            return response()->json(['data' => $types]);
        } catch (\Exception $e) {
            Log::error('NotificationMobile getNotificationTypes error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت انواع نوتیفیکیشن'], 500);
        }
    }
}
