<?php

namespace App\Http\Controllers\V1\Admin;

use App\Enums\OrderStatus;
use App\Enums\TransactionType;
use App\Http\Controllers\Controller;
use App\Repositories\Branch\BranchRepository;
use App\Repositories\Order\OrderRepository;
use App\Repositories\Transaction\TransactionRepository;
use App\Repositories\User\UserRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Enum;

class OrderController extends Controller
{
    protected $orderRepository;

    public function __construct()
    {
        $this->orderRepository = app(OrderRepository::class);
    }

    public function index(Request $request)
    {
        $categories = $this->orderRepository->getWithFilter($request->all());
        return response()->json(['data' => $categories], 200);
    }

    public function historyOrder(Request $request)
    {
        $categories = $this->orderRepository->getWithFilter($request->all());
        return response()->json(['data' => $categories], 200);
    }
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric|exists:orders,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $order = $this->orderRepository->showA($request->order_id);
        if (!$order) {
            return response()->json(['message' => 'دسته بندی یافت نشد'], 404);
        }

        return response()->json(['data' => $order]);
    }

    public function changeStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric|exists:orders,id',
            'status' => [new Enum(OrderStatus::class)],
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $order = $this->orderRepository->changeStatus($request->order_id, $request->status);

        return response()->json($order, $order['status']);
    }

    public function acceptOrder(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric|exists:orders,id',
            'status' => [new Enum(OrderStatus::class)],
            'preparation_time' => 'required|numeric',
        ]);
        if ($validator->fails()) {
            return response()->json(['status' => '400', 'data' => [], 'message' => $validator->errors()->first()], 400);
        }
        return $this->orderRepository->acceptOrder($request->order_id, (int) $request->preparation_time);
    }

    public function getOrderForSend(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric|exists:orders,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $order = $this->orderRepository->getOrderForSend($request->order_id);
        if (!$order) {
            return response()->json(['message' => 'دسته بندی یافت نشد'], 404);
        }
        return response()->json(['message' => 'با موفقیت انجام شد']);
    }

    public function setDeliveryOrder(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric|exists:orders,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $order = $this->orderRepository->setDeliveryOrder($request->order_id);
        return response()->json($order, $order['status'] ?? 500);
    }

    public function orderReturned(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric|exists:orders,id',
            'amount' => 'required',
            'description' => 'required',
            'whoseAccount' => 'sometimes',
        ]);        
    
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
    
        DB::beginTransaction(); // شروع تراکنش
    
        try {
            $order = $this->orderRepository->show($request->order_id);
            $userBranch = app(UserRepository::class)->getUserWithBranchId($order->branch_id);
            $orderPrice = $order->order_price;
            $deliveryPrice = $order->delivery_price;
            $transactionRep = app(TransactionRepository::class);
    
            // اگر سفارش تحویل شده
            if ($order->order_status == 'delivered') {
                switch ($request->whoseAccount) {
                    case 'branch':
                        $transactionRep->storeP([
                            'amount' => $orderPrice * -1,
                            'user_id' => $userBranch->id,
                            'transaction_type' => TransactionType::RETURNED,
                            'description' => $request->description,
                            'target_id' => $order->id
                        ]);
                        break;
    
                    case 'driver':
                        $transactionRep->storeP([
                            'amount' => $orderPrice * -1,
                            'user_id' => $order->driver_id,
                            'transaction_type' => TransactionType::RETURNED,
                            'description' => $request->description,
                            'target_id' => $order->id
                        ]);
                        break;
    
                    case 'branchAndDriver':
                        $transactionRep->storeP([
                            'amount' => ($orderPrice - $deliveryPrice) * -1,
                            'user_id' => $userBranch->id,
                            'transaction_type' => TransactionType::RETURNED,
                            'description' => $request->description,
                            'target_id' => $order->id
                        ]);
                        $transactionRep->storeP([
                            'amount' => $deliveryPrice * -1,
                            'user_id' => $order->driver_id,
                            'transaction_type' => TransactionType::RETURNED,
                            'description' => $request->description,
                            'target_id' => $order->id
                        ]);
                        break;
    
                    case 'system':
                        $transactionRep->storeP([
                            'amount' => $orderPrice * -1,
                            'user_id' => 0,
                            'transaction_type' => TransactionType::RETURNEDSYSTEM,
                            'target_id' => $order->id,
                            'description' => $request->description
                        ]);
                        break;
                }
                app(OrderRepository::class)->update($order->id, ['order_status' => OrderStatus::RETURNED]);
            } 
            // اگر سفارش در انتظار پرداخت یا در حال سفارش نیست
            elseif ($order->order_status != OrderStatus::WATINGPAYMENT && $order->order_status != OrderStatus::ORDERING) {
                $sumPrice = $order->order_price;
    
                if ($order->cash_back_price > 0) {
                    $sumPrice -= $order->cash_back_price;
                    app(UserRepository::class)->setWalletBranch($order->branch_id, $order->user_id, $order->cash_back_price, null);
                }
    
                if ($order->wallet_use_price > 0) {
                    $sumPrice -= $order->wallet_use_price;
                }
    
                if ($order->type_order == 'platform') {
                    Log::info($sumPrice + $order->wallet_use_price);
                    $transactionRep->storeP([
                        'amount' => $sumPrice + $order->wallet_use_price,
                        'user_id' => $order->user_id,
                        'transaction_type' => TransactionType::CANCELLED,
                        'target_id' => $order->id,
                        'description' => $request->description
                    ]);
                } else {
                    if ($order->order_price_payed > 0) {
                        $sumPrice -= $order->order_price_payed;
                    }
                    $transactionRep->storeP([
                        'amount' => $sumPrice,
                        'user_id' => $order->user_id,
                        'transaction_type' => TransactionType::CANCELLED,
                        'target_id' => $order->id,
                        'description' => $request->description
                    ]);
                }
    
                app(OrderRepository::class)->update($order->id, ['order_status' => OrderStatus::CANCELLED]);
            }
    
            DB::commit(); // ذخیره تراکنش
            return response()->json(['message' => 'با موفقیت انجام شد']);
        } catch (\Throwable $e) {
            DB::rollBack(); // rollback در صورت بروز خطا
            Log::error("Error in orderReturned: " . $e->getMessage());
            return response()->json([
                'message' => 'عملیات با شکست مواجه شد: ' . $e->getMessage()
            ], 500);
        }
    }

    public function orderUpdate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric',
            'order_data' => 'required|array',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $paymentType = $request->input('payment_type', null);
        $payFromWallet = $request->input('pay_from_wallet', 0);
        $payOther = $request->input('pay_other', 0);

        // get now order
        $order = $this->orderRepository->show($request->order_id);
        if (!$order) {
            return response()->json(['error' => 'سفارش یافت نشد'], 404);
        }

        $oldOrderPrice = 0;
        foreach ($order->carts as $cart) {
            if (isset($cart['variety'])) {
                $price = isset($cart['variety']['discounted_price']) ? (int) $cart['variety']['discounted_price'] : (int) $cart['variety']['price'];
                $count = isset($cart['count']) ? (int) $cart['count'] : 1;
                $oldOrderPrice += $price * $count;
            } else {
                $price = isset($cart['discounted_price']) ? (int) $cart['discounted_price'] : (int) $cart['price'];
                $count = isset($cart['count']) ? (int) $cart['count'] : 1;
                $oldOrderPrice += $price * $count;
            }
        }

        $orderData = $request->order_data;
        $newOrderPrice = 0;
        foreach ($orderData as $item) {
            if (isset($item['variety'])) {
                $price = isset($item['variety']['discounted_price']) ? (int) $item['variety']['discounted_price'] : (int) $item['variety']['price'];
                $count = isset($item['count']) ? (int) $item['count'] : 1;
                $newOrderPrice += $price * $count;
            } else {
                $price = isset($item['discounted_price']) ? (int) $item['discounted_price'] : (int) $item['price'];
                $count = isset($item['count']) ? (int) $item['count'] : 1;
                $newOrderPrice += $price * $count;
            }
        }
        $diff = $newOrderPrice - $oldOrderPrice;

        $paymentLink = null;
        $transactionRep = app(\App\Repositories\Transaction\TransactionRepository::class);

        if ($diff > 0) {
            if ($paymentType === 'cash') {
                $this->orderRepository->update($request->order_id, [
                    'order_status' => 'preparing',
                    'payment_status' => 'true',
                ]);
            } elseif ($paymentType === 'wallet') {
                $userId = $order->user_id;

                $walletBalance = $transactionRep->userInventoryWithUser($userId);

                if ($walletBalance >= $diff) {
                    // موجودی کافی است - تمام مبلغ از کیف پول کسر می‌شود
                    $transactionRep->storeP([
                        'amount' => -$diff,
                        'user_id' => $userId,
                        'transaction_type' => \App\Enums\TransactionType::PAYORDER,
                        'description' => 'پرداخت اختلاف ویرایش سفارش از کیف پول',
                        'target_id' => $order->id,
                    ]);
                    $this->orderRepository->update($request->order_id, [
                        'order_status' => 'preparing',
                        'payment_status' => 'true',
                    ]);
                } else {
                    return response()->json(['error' => 'موجودی کیف پول کافی نیست. موجودی: ' . number_format($walletBalance) . ' تومان، مبلغ مورد نیاز: ' . number_format($diff) . ' تومان'], 400);
                }
            } elseif ($paymentType === 'mixed') {
                $userId = $order->user_id;
                $walletBalance = $transactionRep->userInventoryWithUser($userId);
                $remainingPaymentType = $request->input('remaining_payment_type', 'cash');

                if ($payFromWallet > 0 && $walletBalance >= $payFromWallet) {
                    $transactionRep->storeP([
                        'amount' => -$payFromWallet,
                        'user_id' => $userId,
                        'transaction_type' => \App\Enums\TransactionType::PAYORDER,
                        'description' => 'پرداخت بخشی از اختلاف ویرایش سفارش از کیف پول',
                        'target_id' => $order->id,
                    ]);

                    if ($payOther > 0) {
                        if ($remainingPaymentType === 'cash') {
                            $this->orderRepository->update($request->order_id, [
                                'order_status' => 'preparing',
                                'payment_status' => 'true',
                            ]);
                        } else {
                            $paymentLink = 'لینک پرداخت برای مبلغ باقی‌مانده: ' . number_format($payOther) . ' تومان';
                            $this->orderRepository->update($request->order_id, [
                                'order_status' => 'wating_payment',
                                'payment_status' => 'false',
                            ]);
                        }
                    } else {
                        $this->orderRepository->update($request->order_id, [
                            'order_status' => 'preparing',
                            'payment_status' => 'true',
                        ]);
                    }
                } else {
                    return response()->json(['error' => 'موجودی کیف پول کافی نیست یا مبلغ نامعتبر است'], 400);
                }
            } else {
                $paymentLink = 'لینک پرداخت برای مبلغ: ' . number_format($diff) . ' تومان';
                $this->orderRepository->update($request->order_id, [
                    'order_status' => 'wating_payment',
                    'payment_status' => 'false',
                ]);
            }
        } elseif ($diff < 0) {
            $userId = $order->user_id;
            $transactionRep->storeP([
                'amount' => abs($diff),
                'user_id' => $userId,
                'transaction_type' => \App\Enums\TransactionType::CHARGE,
                'description' => 'برگشت اختلاف ویرایش سفارش',
                'target_id' => $order->id,
            ]);
        }

        $result = $this->orderRepository->updateOrderAndCartsWithHistory($request->order_id, $orderData, $diff);

        if ($paymentLink) {
            $result['payment_link'] = $paymentLink;
        }
        $status = is_array($result) && isset($result['status']) ? $result['status'] : 500;

        return response()->json(['message' => 'عملیات با موفقیت انجام شد', 'statusCode' => 200], 200);
    }
}
