<?php

namespace App\Http\Controllers\V1\Admin;

use App\Http\Controllers\Controller;
use App\Repositories\PrinterSetting\PrinterSettingRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class PrinterSettingController extends Controller
{
    protected $printerSettingRepository;

    public function __construct()
    {
        $this->printerSettingRepository = app(PrinterSettingRepository::class);
    }

    public function index(Request $request)
    {
        try {
            $printerSettings = $this->printerSettingRepository->getWithFilter($request->all(), true);
            return response()->json(['data' => $printerSettings]);
        } catch (\Exception $e) {
            Log::error('PrinterSetting index error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت تنظیمات پرینتر'], 500);
        }
    }

    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'printer_setting_id' => 'required|numeric|exists:printer_settings,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            $printerSetting = $this->printerSettingRepository->show($request->printer_setting_id);

            if (!$printerSetting) {
                return response()->json(['message' => 'تنظیمات پرینتر یافت نشد'], 404);
            }

            return response()->json(['data' => $printerSetting]);
        } catch (\Exception $e) {
            Log::error('PrinterSetting show error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت تنظیمات پرینتر'], 500);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
            'name' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'options' => 'required|json',
            'status' => 'nullable|in:active,inactive',
            'type' => 'required|in:app,qz',
            'size' => 'required|in:6cm,8cm,10cm'
        ], [
            'branch_id.required' => 'شعبه الزامی است',
            'branch_id.exists' => 'شعبه انتخابی معتبر نیست',
            'name.required' => 'نام الزامی است',
            'name.string' => 'نام باید متن باشد',
            'name.max' => 'نام نباید بیشتر از 255 کاراکتر باشد',
            'title.required' => 'عنوان الزامی است',
            'title.string' => 'عنوان باید متن باشد',
            'title.max' => 'عنوان نباید بیشتر از 255 کاراکتر باشد',
            'options.required' => 'تنظیمات الزامی است',
            'options.json' => 'تنظیمات باید در فرمت JSON باشد',
            'status.in' => 'وضعیت باید active یا inactive باشد',
            'type.required' => 'نوع الزامی است',
            'type.in' => 'نوع باید app یا qz باشد'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            // Validate JSON format
            $options = json_decode($request->options, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                return response()->json(['error' => 'فرمت JSON تنظیمات نامعتبر است'], 400);
            }

            DB::beginTransaction();

            $data = [
                'branch_id' => $request->branch_id,
                'name' => $request->name,
                'title' => $request->title,
                'options' => $request->options,
                'status' => $request->status ?? 'active',
                'type' => $request->type,
                'size' => $request->size
            ];

            $printerSetting = $this->printerSettingRepository->store($data);

            if (!$printerSetting) {
                DB::rollBack();
                return response()->json(['message' => 'خطا در ایجاد تنظیمات پرینتر'], 400);
            }

            DB::commit();

            return response()->json([
                'data' => $printerSetting,
                'message' => 'تنظیمات پرینتر با موفقیت ایجاد شد'
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('PrinterSetting store error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در ایجاد تنظیمات پرینتر'], 500);
        }
    }

    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'printer_setting_id' => 'required|numeric|exists:printer_settings,id',
            'branch_id' => 'required|exists:branches,id',
            'name' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'options' => 'required|json',
            'status' => 'nullable|in:active,inactive',
            'type' => 'required|in:app,qz',
            'size' => 'required|in:6cm,8cm,10cm'
        ], [
            'printer_setting_id.required' => 'شناسه تنظیمات پرینتر الزامی است',
            'printer_setting_id.exists' => 'تنظیمات پرینتر یافت نشد',
            'branch_id.required' => 'شعبه الزامی است',
            'branch_id.exists' => 'شعبه انتخابی معتبر نیست',
            'name.required' => 'نام الزامی است',
            'name.string' => 'نام باید متن باشد',
            'name.max' => 'نام نباید بیشتر از 255 کاراکتر باشد',
            'title.required' => 'عنوان الزامی است',
            'title.string' => 'عنوان باید متن باشد',
            'title.max' => 'عنوان نباید بیشتر از 255 کاراکتر باشد',
            'options.required' => 'تنظیمات الزامی است',
            'options.json' => 'تنظیمات باید در فرمت JSON باشد',
            'status.in' => 'وضعیت باید active یا inactive باشد',
            'type.required' => 'نوع الزامی است',
            'type.in' => 'نوع باید app یا qz باشد'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            // Validate JSON format
            $options = json_decode($request->options, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                return response()->json(['error' => 'فرمت JSON تنظیمات نامعتبر است'], 400);
            }

            DB::beginTransaction();

            $data = [
                'branch_id' => $request->branch_id,
                'name' => $request->name,
                'title' => $request->title,
                'options' => $request->options,
                'status' => $request->status ?? 'active',
                'type' => $request->type,
                'size' => $request->size
            ];

            $result = $this->printerSettingRepository->update($request->printer_setting_id, $data);

            if (!$result) {
                DB::rollBack();
                return response()->json(['message' => 'خطا در بروزرسانی تنظیمات پرینتر'], 400);
            }

            DB::commit();

            // Get updated record
            $updatedPrinterSetting = $this->printerSettingRepository->show($request->printer_setting_id);

            return response()->json([
                'data' => $updatedPrinterSetting,
                'message' => 'تنظیمات پرینتر با موفقیت بروزرسانی شد'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('PrinterSetting update error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در بروزرسانی تنظیمات پرینتر'], 500);
        }
    }

    public function destroy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'printer_setting_id' => 'required|numeric|exists:printer_settings,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            DB::beginTransaction();

            $result = $this->printerSettingRepository->destroy($request->printer_setting_id);

            if (!$result) {
                DB::rollBack();
                return response()->json(['message' => 'خطا در حذف تنظیمات پرینتر'], 400);
            }

            DB::commit();

            return response()->json(['message' => 'تنظیمات پرینتر با موفقیت حذف شد']);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('PrinterSetting destroy error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در حذف تنظیمات پرینتر'], 500);
        }
    }

    public function getByBranch(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|numeric|exists:branches,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            $printerSettings = $this->printerSettingRepository->getByBranch($request->branch_id);
            return response()->json(['data' => $printerSettings]);
        } catch (\Exception $e) {
            Log::error('PrinterSetting getByBranch error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت تنظیمات پرینتر شعبه'], 500);
        }
    }

    public function getByType(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'type' => 'required|in:app,qz',
            'branch_id' => 'nullable|numeric|exists:branches,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            $printerSettings = $this->printerSettingRepository->getByType($request->type, $request->branch_id);
            return response()->json(['data' => $printerSettings]);
        } catch (\Exception $e) {
            Log::error('PrinterSetting getByType error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت تنظیمات پرینتر بر اساس نوع'], 500);
        }
    }
}
