<?php

namespace App\Http\Controllers\V1\Admin;

use App\Enums\ProductStatus;
use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductMaterial;
use App\Repositories\Product\ProductRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Enum;

class ProductController extends Controller
{

    protected $productRepository;

    public function __construct()
    {
        $this->productRepository = app(ProductRepository::class);
    }

    public function index(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'nullable|numeric|exists:branches,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $product = $this->productRepository->getWithFilter($request->all(), true);
        return response()->json(['data' => $product]);
    }

    public function store(Request $request)
    {
        $category = $request->category_id;
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'branch_id' => 'required|numeric|exists:branches,id',
            'image' => 'nullable|url',
            'menu_id' => 'required|numeric|exists:categories,id',
            'category_id' => [Rule::requiredif(function () use ($category) {
                return $category == 0 ? "" : "exists:categories,id";
            }), 'required', 'numeric'],
            'description' => 'nullable'
        ]);
        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->first(), 'status' => 400, 'data' => []], 400);
        }
        $product = $this->productRepository->storeP($request->all());
        if ($product) {
            return response()->json(['data' => $product, 'message' => '', 'status' => 201], 201);
        }
        return response()->json(['message' => 'عملیات با شکست مواجه شد'], 400);


    }

    public function storeWithIdsDefault(Request $request)
    {
        $category = $request->category_id;
        $validator = Validator::make($request->all(), [
            'ids' => 'required|array|min:1',
            'ids.*' => [Rule::requiredif(function () use ($category) {
                return $category == 0 ? "" : "exists:product_defaults,id";
            }), 'required', 'numeric'],
        ]);
        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->first(), 'status' => 400, 'data' => []], 400);
        }
        $product = $this->productRepository->storeWithIdsDefault($request->all());
        if ($product) {
            return response()->json(['data' => $product, 'message' => '', 'status' => 200], 200);
        }
        return response()->json(['message' => 'عملیات با شکست مواجه شد'], 400);
    }

    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'product_id' => 'required|numeric|exists:products,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $product = $this->productRepository->showWithRel($request->product_id);

        if (!$product) {
            return response()->json(['message' => 'هیچ موردی یافت نشد'], 404);
        }
        return response()->json(['data' => $product]);
    }

    public function update(Request $request)
    {
        $category = $request->category_id;
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'product_id' => 'required|numeric|exists:products,id',
            'branch_id' => 'nullable|numeric|exists:branches,id',
            'menu_id' => 'required|numeric|exists:categories,id',
            'category_id' => [Rule::requiredif(function () use ($category) {
                return $category == 0 ? "" : "exists:categories,id";
            }), 'required', 'numeric'],
            'image' => 'nullable|url',
            'description' => 'nullable',
            'status' => [new Enum(ProductStatus::class)],

        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $product = $this->productRepository->update($request->product_id, $request->all());
        if ($product) {
            return response()->json(['message' => 'عملیات با موفقیت انجام شد'], 200);
        }
        return response()->json(['message' => 'عملیات با شکست مواجه شد'], 400);
    }

    public function destroy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'product_id' => 'required|numeric|exists:products,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $product = $this->productRepository->destroy($request->product_id);
        if (!$product) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 404);
        }
        return response()->json(['message' => 'با موفقیت حذف شد '], 200);
    }

    public function updateCountVariety(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'variety_id' => 'nullable|numeric|exists:varieties,id',
            'count' => 'required|numeric',
        ]);
        if ($validator->fails()) {
            return response()->json(['status' => 400, 'data' => [], 'message' => $validator->errors()->first()], 400);
        }
        $variety = $this->productRepository->updateCountVariety($request->all());
        return response()->json($variety, $variety['status']);
    }


    public function updateOrStoreVarieties(Request $request, $productId)
    {


        $validator = Validator::make($request->all(), [
            'varieties' => 'required',
            'varieties.*.title' => [
                'required',
                'min:3',
            ],
            'varieties.*.price' => [
                'required',
                'numeric',
            ],
            'varieties.*.price_paking' => [
                'required',
                'numeric',
            ],
            'varieties.*.discount' => [
                'required',
                'numeric',
                'max:100',
                'min:0'
            ],
            'varieties.*.count' => [
                'required',
                'numeric',
                'min:0'
            ],
            'varieties.*.max_order' => [
                'required',
                'numeric',
                'min:1'
            ],
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 400, 'data' => [], 'message' => $validator->errors()->first()], 400);
        }

        $data = $this->productRepository->updateOrStoreVarieties($request->varieties ?? [], $productId);
        return response()->json($data, $data['status']);
    }


    public function setGallery(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'product_id' => 'required|numeric|exists:products,id',
            'images' => 'nullable|array',
            'images.*' => 'nullable|url',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $product = $this->productRepository->show($request->product_id);

        if ($request->images) {
            $array = [];
            foreach ($request->images as $img) {
                $array[] = ['path' => $img, 'user_id' => auth()->id(), 'code' => Str::random(32)];
            }
            $product->update(['images' => json_encode($array)]);
        } else
            $product->update(['images' => null]);

        return response()->json(['data' => $product], 200);
    }

    public function setMaterials(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'product_id' => 'required|numeric|exists:products,id',
                'materials' => 'required|array',
                'materials.*.id' => 'required|numeric|exists:materials,id',

            ]);
            if ($validator->fails()) {
                return response()->json(['error' => $validator->errors()->first()], 400);
            }
            $products = $this->productRepository->show($request->product_id);
            ProductMaterial::where('product_id', $request->product_id)->delete();
            $materials = $request->materials;
            $mat = [];
            foreach ($materials as $material)
                $mat[] = ['product_id' => $request->product_id, 'material_id' => $material['id'], 'usage_per_unit' => $material['used']];

            DB::table('product_materials')->insert($mat);

            $calories = app(ProductRepository::class)->getAllCalories($request->product_id);
            Product::where('id', $request->product_id)->update(['calory' => $calories['data']['energy']]);

            return response()->json(['data' => true], 200);

        } catch (\Exception $e) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 400);
        }
    }

}
