<?php

namespace App\Http\Controllers\V1\Admin;

use App\Enums\ProductStatus;
use App\Http\Controllers\Controller;
use App\Imports\ProductDefaultImport;
use App\Models\ProductDefault;
use App\Repositories\ProductDefault\ProductDefaultRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Enum;

class ProductDefaultController extends Controller
{

    protected $productRepository;

    public function __construct()
    {
        $this->productRepository = app(ProductDefaultRepository::class);
    }

    public function index(Request $request)
    {
        $product = $this->productRepository->getWithFilter($request->all(), true);
        return response()->json(['data' => $product]);
    }

    public function store(Request $request)
    {
        $category = $request->category_id;
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'image' => 'nullable|url',
            'category_id' => [Rule::requiredif(function () use ($category) {
                return $category == 0 ? "" : "exists:categories,id";
            }), 'required', 'numeric'],
            'description' => 'nullable'
        ]);
        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->first(), 'status' => 400, 'data' => []], 400);
        }
        $product = $this->productRepository->storeP($request->all());
        if ($product) {
            return response()->json(['data' => $product, 'message' => '', 'status' => 201], 201);
        }
        return response()->json(['message' => 'عملیات با شکست مواجه شد'], 400);


    }

    public function show(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'product_id' => 'required|numeric|exists:product_defaults,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $product = $this->productRepository->show($request->product_id);

        if (!$product) {
            return response()->json(['message' => 'هیچ موردی یافت نشد'], 404);
        }
        return response()->json(['data' => $product]);
    }

    public function update(Request $request)
    {
        $category = $request->category_id;
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'product_id' => 'required|numeric|exists:product_defaults,id',
            'category_id' => [Rule::requiredif(function () use ($category) {
                return $category == 0 ? "" : "exists:categories,id";
            }), 'required', 'numeric'],
            'image' => 'nullable|url',
            'description' => 'nullable',
            'status' => [new Enum(ProductStatus::class)],
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $product = $this->productRepository->update($request->product_id, $request->all());
        if ($product) {
            return response()->json(['message' => 'عملیات با موفقیت انجام شد'], 200);
        }
        return response()->json(['message' => 'عملیات با شکست مواجه شد'], 400);
    }

    public function destroy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'product_id' => 'required|numeric|exists:product_defaults,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $product = $this->productRepository->destroy($request->product_id);
        if (!$product) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 404);
        }
        return response()->json(['message' => 'با موفقیت حذف شد '], 200);
    }


    public function setGallery(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'product_id' => 'required|numeric|exists:product_defaults,id',
            'images' => 'required|array',
            'images.*' => 'required|url',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $product = $this->productRepository->show($request->product_id);
        $array = [];
        foreach ($request->images as $img) {
            $array[] = ['path' => $img, 'user_id' => auth()->id(), 'code' => Str::random(32)];
        }
        $product->update(['images' => json_encode($array)]);
        return response()->json(['data' => $product], 200);
    }

    public function productDefaultImport()
    {
        \Maatwebsite\Excel\Facades\Excel::import(new ProductDefaultImport(), 'Excel/Import/products.xls');
    }
}
