<?php

namespace App\Http\Controllers\V1\Admin;

use App\Http\Controllers\Controller;
use App\Repositories\SmsSetting\SmsSettingRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class SmsSettingController extends Controller
{
    protected $smsSettingRepository;

    public function __construct()
    {
        $this->smsSettingRepository = app(SmsSettingRepository::class);
    }

    public function index(Request $request)
    {
        try {
            $smsSettings = $this->smsSettingRepository->getWithFilter($request->all());
            return response()->json(['data' => $smsSettings]);
        } catch (\Exception $e) {
            Log::error('SmsSetting index error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت تنظیمات پیامک'], 500);
        }
    }

    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'sms_setting_id' => 'required|numeric|exists:sms_settings,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            $smsSetting = $this->smsSettingRepository->show($request->sms_setting_id);

            if (!$smsSetting) {
                return response()->json(['message' => 'تنظیمات پیامک یافت نشد'], 404);
            }

            return response()->json(['data' => $smsSetting]);
        } catch (\Exception $e) {
            Log::error('SmsSetting show error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت تنظیمات پیامک'], 500);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'repository' => 'nullable|string|max:255',
            'status' => 'nullable|boolean'
        ], [
            'title.required' => 'عنوان الزامی است',
            'title.string' => 'عنوان باید متن باشد',
            'title.max' => 'عنوان نباید بیشتر از 255 کاراکتر باشد',
            'repository.string' => 'مخزن باید متن باشد',
            'repository.max' => 'مخزن نباید بیشتر از 255 کاراکتر باشد',
            'status.boolean' => 'وضعیت باید true یا false باشد'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            DB::beginTransaction();

            $data = [
                'title' => $request->title,
                'status' => $request->status ?? true,
                'order' => $request->order ?? 0
            ];
            $smsSetting = $this->smsSettingRepository->store($data);

            if (!$smsSetting) {
                DB::rollBack();
                return response()->json(['message' => 'خطا در ایجاد تنظیمات پیامک'], 400);
            }

            DB::commit();

            return response()->json([
                'data' => $smsSetting,
                'message' => 'تنظیمات پیامک با موفقیت ایجاد شد'
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('SmsSetting store error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در ایجاد تنظیمات پیامک'], 500);
        }
    }

    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'sms_setting_id' => 'required|numeric|exists:sms_settings,id',
            'title' => 'required|string|max:255',
            'repository' => 'nullable|string|max:255',
            'status' => 'nullable|boolean'
        ], [
            'sms_setting_id.required' => 'شناسه تنظیمات پیامک الزامی است',
            'sms_setting_id.exists' => 'تنظیمات پیامک یافت نشد',
            'title.required' => 'عنوان الزامی است',
            'title.string' => 'عنوان باید متن باشد',
            'title.max' => 'عنوان نباید بیشتر از 255 کاراکتر باشد',
            'repository.string' => 'مخزن باید متن باشد',
            'repository.max' => 'مخزن نباید بیشتر از 255 کاراکتر باشد',
            'status.boolean' => 'وضعیت باید true یا false باشد'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            DB::beginTransaction();

            $data = [
                'title' => $request->title,
                'status' => $request->status ?? true,
                'order' => $request->order ?? 0
            ];

            $result = $this->smsSettingRepository->update($request->sms_setting_id, $data);

            if (!$result) {
                DB::rollBack();
                return response()->json(['message' => 'خطا در بروزرسانی تنظیمات پیامک'], 400);
            }

            DB::commit();

            // Get updated record
            $updatedSmsSetting = $this->smsSettingRepository->show($request->sms_setting_id);

            return response()->json([
                'data' => $updatedSmsSetting,
                'message' => 'تنظیمات پیامک با موفقیت بروزرسانی شد'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('SmsSetting update error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در بروزرسانی تنظیمات پیامک'], 500);
        }
    }

    public function destroy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'sms_setting_id' => 'required|numeric|exists:sms_settings,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            DB::beginTransaction();

            $result = $this->smsSettingRepository->destroy($request->sms_setting_id);

            if (!$result) {
                DB::rollBack();
                return response()->json(['message' => 'خطا در حذف تنظیمات پیامک'], 400);
            }

            DB::commit();

            return response()->json(['message' => 'تنظیمات پیامک با موفقیت حذف شد']);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('SmsSetting destroy error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در حذف تنظیمات پیامک'], 500);
        }
    }

    public function getByRepository(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'repository' => 'required|string'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        try {
            $smsSettings = $this->smsSettingRepository->getWithFilter(['repository' => $request->repository]);
            return response()->json(['data' => $smsSettings]);
        } catch (\Exception $e) {
            Log::error('SmsSetting getByRepository error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت تنظیمات پیامک بر اساس مخزن'], 500);
        }
    }

    public function getActiveSettings(Request $request)
    {
        try {
            $smsSettings = $this->smsSettingRepository->getWithFilter(['status' => true]);
            return response()->json(['data' => $smsSettings]);
        } catch (\Exception $e) {
            Log::error('SmsSetting getActiveSettings error: ' . $e->getMessage());
            return response()->json(['message' => 'خطا در دریافت تنظیمات فعال پیامک'], 500);
        }
    }
}
