<?php

namespace App\Http\Controllers\V1;

use App\Http\Controllers\Controller;
use App\Models\Area;
use App\Repositories\Area\AreaRepository;
use App\Repositories\AreaManagment\AreaManagmentRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use App\Services\AreaManagementService;
use App\Services\AreaValidationService;
use App\Http\Requests\AreaManagementRequest;
use App\Models\User;
use App\Repositories\Order\OrderRepository;
use App\Models\Order;
use App\Models\Branch;
use App\Helpers\PointInPolygonHelper;
use App\Models\Setting;
use App\Repositories\Address\AddressRepository;
use App\Repositories\Branch\BranchRepository;
use App\Repositories\BranchHoliday\BranchHolidayRepository;
use App\Repositories\Delivery\DeliveryRepository;
use App\Repositories\Delivery\DeliveryType\ComponentByAreaManagement;
use App\Repositories\Setting\SettingRepository;
use Illuminate\Support\Facades\Log;

class AreaController extends Controller
{
    protected $areaManagmentRepository;
    protected $orderRep;
    protected $holidayRepository;

    public function __construct(
        AreaManagmentRepository $areaManagmentRepository,
        OrderRepository $orderRepository,
        BranchHolidayRepository $holidayRepository
    ) {
        $this->areaManagmentRepository = $areaManagmentRepository;
        $this->orderRep = $orderRepository;
        $this->holidayRepository = $holidayRepository;
    }

    public function getArea(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'area_id' => 'nullable|numeric|exists:areas,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        if ($request->parent_id and $request->parent_id !== null)
            $parentId = $request->parent_id;
        else
            $parentId = 0;

        $areas = app(AreaRepository::class)->getWithParentId($parentId);

        return response()->json(['data' => $areas]);

    }


    public function getActiveShippingMethods(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|numeric|exists:branches,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $branchId = $request->branch_id;
        $now = Carbon::now();
        $currentDay = $now->format('l');
        $currentTime = $now->format('H:i:s');
        $areaManagements = app(\App\Repositories\AreaManagment\AreaManagmentRepository::class)
            ->getActiveByBranchDayTime($branchId, $currentDay, $currentTime);

        $shippingMethodIds = [];
        foreach ($areaManagements as $am) {
            if (is_array($am->shipping_methods)) {
                $shippingMethodIds = array_merge($shippingMethodIds, $am->shipping_methods);
            } elseif (is_string($am->shipping_methods)) {
                $decoded = json_decode($am->shipping_methods, true);
                if (is_array($decoded)) {
                    $shippingMethodIds = array_merge($shippingMethodIds, $decoded);
                }
            }
        }
        $shippingMethodIds = array_values(array_unique($shippingMethodIds));


        $deliveryTypes = collect();
        if (!empty($shippingMethodIds)) {
            $deliveryTypes = \App\Models\DeliveryType::whereIn('id', $shippingMethodIds)->get();
        }

        return response()->json(['data' => $deliveryTypes]);
    }


    public function getUserAddressesForDeliveryType(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|numeric|exists:branches,id',
            'delivery_type_id' => 'required|numeric|exists:delivery_types,id',
            'user_id' => 'nullable|numeric|exists:users,id',
            'lat' => 'nullable|numeric',
            'lng' => 'nullable|numeric',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $user = User::find($request->user_id);

        // if (!$user) {
        //     return response()->json(['error' => 'Unauthorized'], 401);
        // }

        $branchId = $request->branch_id;
        $deliveryTypeId = $request->delivery_type_id;

        $this->updateOrder(['delivery_type_id' => $deliveryTypeId]);

        $deliveryType = \App\Models\DeliveryType::find($deliveryTypeId);
        if ($deliveryType && in_array($deliveryType->delivery_method, ['serve', 'actual'])) {

            $token = null;
            if (isset($request->guest_token)) {
                $token = $request->guest_token;
            }

            $order = app(OrderRepository::class)->getLatestOrderWithItems($token, $deliveryTypeId);
            return response()->json(['data' => [], 'order' => $order]);
        }



        $setting = Setting::where('name', 'shippingType')->where('branch_id', null)->latest()->first();
        $typeDelivery = $setting ? $setting->value : 'withArea';


        if ($typeDelivery=="withArea"){
        $now = Carbon::now();
        $currentDay = $now->format('l');
        $currentTime = $now->format('H:i:s');

        $areaManagements = app(\App\Repositories\AreaManagment\AreaManagmentRepository::class)
            ->getActiveByBranchDayTime($branchId, $currentDay, $currentTime);


        $areaIds = [];
        foreach ($areaManagements as $am) {
            $shippingMethods = is_array($am->shipping_methods) ? $am->shipping_methods : json_decode($am->shipping_methods, true);
            if (is_array($shippingMethods) && in_array($deliveryTypeId, $shippingMethods)) {
                $areaIds[] = $am->area_id;
            }
        }

        $areaIds = array_unique($areaIds);
        if (empty($areaIds)) {
            return response()->json(['data' => [], 'is_location_in_active_area' => false]);
        }

        $areas = \App\Models\Area::whereIn('id', $areaIds)->get(['id', 'coordinates']);
        $areasById = $areas->keyBy('id');

        $isLocationInActiveArea = null;
        if ($request->has('lat') && $request->has('lng')) {
            $lat = $request->lat;
            $lng = $request->lng;
            $isLocationInActiveArea = false;
            foreach ($areasById as $area) {
                if (self::pointInPolygon([$lat, $lng], $area->coordinates)) {
                    $isLocationInActiveArea = true;
                    break;
                }
            }
        }

        if($user){
            $addresses = $user->address()->get();
            $resultAddresses = [];

            foreach ($addresses as $address) {
                $activeAreaId = null;
                if (isset($address->lat) && isset($address->lng)) {
                    foreach ($areasById as $area) {
                        if (self::pointInPolygon([$address->lat, $address->lng], $area->coordinates)) {
                            $activeAreaId = $area->id;
                            break;
                        }
                    }
                }
                $addressData = $address->toArray();
                $addressData['active_area_id'] = $activeAreaId;
                $addressData['is_active'] = $activeAreaId !== null;
                $resultAddresses[] = $addressData;
            }
        }
            return response()->json(['data' => $resultAddresses ?? [], 'is_location_in_active_area' => $isLocationInActiveArea]);
        }else{
            $settingRep = app(\App\Repositories\Setting\SettingRepository::class);
            $settingShipment = $settingRep->getSettingShippingTypeAndPrice();



            $branch=app(BranchRepository::class)->show($branchId);
            //{"data":[],"is_location_in_active_area":false}
            //checkAccessAddress




            $point1 = [
                'latitude' => $branch->lat,
                'longitude' => $branch->lng,
            ];
            if($user){
                $addresses = $user->address()->get();
                $resultAddresses = [];

                foreach ($addresses as $address) {
                    $activeAreaId = null;
                    $point2 = [
                        'latitude' => $address->lat,
                        'longitude' => $address->lng,
                    ];
                    $distance = haversineDistance($point1, $point2);
                    $active=true;
                    if($distance > $settingShipment['fixedShippingKilometer'])
                      $active=false;

                    $addressData = $address->toArray();
                    $addressData['active_area_id'] = 0;
                    $addressData['is_active'] = $active;
                    $resultAddresses[] = $addressData;
                }
                return response()->json(['data' => $resultAddresses ?? [], 'is_location_in_active_area' => true]);

            }




            $point2 = [
                'latitude' => $request->lat,
                'longitude' => $request->lng,
            ];
            $distance = haversineDistance($point1, $point2);
            if (isset($settingShipment['deliveryFreeKilometer']) and $distance < $settingShipment['deliveryFreeKilometer'])
                return response()->json(['data' => ['cost' => 0], 'is_location_in_active_area' => true]);

            if (isset($settingShipment['fixedShippingKilometer']) and $distance > $settingShipment['fixedShippingKilometer']) {
                $maxKilometer = $distance - $settingShipment['fixedShippingKilometer'];
                $maxPrice = $maxKilometer * $settingShipment['shippingCost'];
                return response()->json(['data' => ['cost' => number_format($settingShipment['fixedShippingCost'] + $maxPrice, 0, '.', '')], 'is_location_in_active_area' => true]);
            }elseif($distance > $settingShipment['fixedShippingKilometer']){
                return response()->json(['data' => [], 'is_location_in_active_area' => false]);
            }
            return response()->json(['data' => ['cost' => number_format($settingShipment['fixedShippingCost'] ?? 0, 0, '.', '')], 'is_location_in_active_area' => true]);


        }
    }


    /**

     * @param array $point [lat, lng]
     * @param array $polygon [[lat, lng], ...]
     * @return bool
     */
    public static function pointInPolygon($point, $polygon)
    {
        if (!is_array($polygon) || count($polygon) < 3) return false;
        $x = $point[0];
        $y = $point[1];
        $inside = false;
        $n = count($polygon);
        for ($i = 0, $j = $n - 1; $i < $n; $j = $i++) {
            $xi = $polygon[$i][0];
            $yi = $polygon[$i][1];
            $xj = $polygon[$j][0];
            $yj = $polygon[$j][1];
            $intersect = (($yi > $y) != ($yj > $y)) &&
                ($x < ($xj - $xi) * ($y - $yi) / (($yj - $yi) ?: 1e-10) + $xi);
            if ($intersect) $inside = !$inside;
        }
        return $inside;
    }


    public function getAreaManagementForAddress(AreaManagementRequest $request, AreaManagementService $service, AreaValidationService $validationService)
    {
        $now = Carbon::now();

        // Check if branch is on holiday
        $holiday = $this->holidayRepository->checkHoliday(
            $now,
            $now->format('H:i:s'),
            $request->branch_id
        );

        if ($holiday) {
            return response()->json(
                $validationService->getHolidayResponse($holiday),
                200
            );
        }

        // Check if branch has any area management records at all
        if (!$this->areaManagmentRepository->hasAnyAreaManagement($request->branch_id)) {
            return response()->json(
                $validationService->getNoAreasRegisteredResponse(),
                200
            );
        }

        if(!$request->address_id){
            return response()->json(
                $validationService->validateAreaWithNoAddress($request),
                200
            );
        }
        $this->updateOrder(['address_id' => $request->address_id]);

        $result = $service->getActiveAreaForAddress(
            $request->address_id,
            $request->branch_id,
            $now->format('l'),
            $now->format('H:i:s'),
            $request->lat,
            $request->lng
        );

        if ($result) {
            return response()->json(
                $validationService->validateAreaAndGetDeliveryInfo($result, $request),
                200
            );
        }

        // Check if there are any active areas for the branch at this time (regardless of user's address)
        $anyActiveArea = $this->areaManagmentRepository->getActiveByBranchDayTime(
            $request->branch_id,
            $now->format('l'),
            $now->format('H:i:s')
        );

        if ($anyActiveArea && $anyActiveArea->isNotEmpty()) {
            // There are active areas, but user's address is not in any of them
            // First try to find area with area_id != 0, if not found return area with area_id = 0
            $areaWithNonZeroId = $anyActiveArea->where('area_id', '!=', 0)->first();
            $areaWithZeroId = $anyActiveArea->where('area_id', 0)->first();

            $selectedArea = $areaWithNonZeroId ?: $areaWithZeroId;

            return response()->json(
                $validationService->validateAreaAndGetDeliveryInfoWithInvalidAddress($selectedArea, $request),
                200
            );
        }

        // Get branch information to check pre-order status
        $branch = Branch::find($request->branch_id);

        // Check if branch has pre-order capability
        if (!$branch || $branch->pre_order_status != 1) {
            return response()->json(
                $validationService->getNearestAvailableTimeResponse(null),
                200
            );
        }



        $nearestData = $this->getNearestAvailableTime($request->branch_id, $now->format('l'), $now->format('H:i:s'));

        return response()->json(
            $validationService->getNearestAvailableTimeResponse($nearestData),
            200
        );
    }
    /**
     * Get nearest available time for branch
     */
    private function getNearestAvailableTime(string $branchId, string $currentDay, string $currentTime)
    {
        return $this->areaManagmentRepository->getNextAvailableByBranch($branchId, $currentDay, $currentTime);
    }

    public function getDeliveryMethodsAndAreaManagementByLocation(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|numeric|exists:branches,id',
            'day' => 'nullable|string',
            'time' => 'nullable|string',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $branchId = $request->branch_id;
        $inputDay = $request->day;
        $inputTime = $request->time;

        $areaManagements = collect();
        $areas = collect();
        $foundAreaIds = [];
        $deliveryTypeIds = [];
        $publicTypeIds = [];

        $branch = Branch::find($branchId);

        if ($inputDay && $inputTime) {
            $currentDay = $inputDay;
            $currentTime = $inputTime;
        } else {
            $now = Carbon::now();
            $currentDay = $now->format('l');
            $currentTime = $now->format('H:i:s');
        }

        $areaManagements = app(\App\Repositories\AreaManagment\AreaManagmentRepository::class)
            ->getActiveByBranchDayTime($branchId, $currentDay, $currentTime);

        $token = null;
        if (isset($request->guest_token))
            $token = $request->guest_token;

        $order = app(OrderRepository::class)->getLatestOrderWithItems($token);
        if($order['order'] instanceof Order && !$areaManagements->isEmpty()){
            $order['order']->update(['area_management_id' => $areaManagements->first()->id, 'area_management' => json_encode($areaManagements->first())]);
        }

        if ($areaManagements->isEmpty()) {
            $nextAvailable = app(\App\Repositories\AreaManagment\AreaManagmentRepository::class)
                ->getNextAvailableByBranch($branchId, $currentDay, $currentTime);
            if ($nextAvailable && $branch->pre_order_status == 1) {
                return response()->json([
                    'error' => 'هیچ مدیریت منطقه فعالی در این لحظه یافت نشد.',
                    'next_available' => [
                        'day' => $nextAvailable->day,
                        'service_time_from' => $nextAvailable->service_time_from,
                        'service_time_to' => $nextAvailable->service_time_to,
                        'shipping_methods' => $nextAvailable->shipping_methods,
                        'area_id' => $nextAvailable->area_id,
                    ]
                ], 200);
            } else {
                return response()->json(['error' => 'هیچ مدیریت منطقه فعالی یافت نشد.'], 200);
            }
        }

        $areaIds = $areaManagements->pluck('area_id')->unique()->toArray();


        if (in_array(0, $areaIds)) {
            foreach ($areaManagements as $am) {
                if($am->area_id == 0){
                $shippingMethods = is_array($am->shipping_methods)
                    ? $am->shipping_methods
                    : json_decode($am->shipping_methods, true);
                if (is_array($shippingMethods)) {
                        $publicTypeIds = array_merge($publicTypeIds, $shippingMethods);
                    }
                }
            }
            $publicTypeIds = array_values(array_unique($publicTypeIds));
            $areaIds = array_diff($areaIds, [0]);
        }

        $areas = \App\Models\Area::whereIn('id', $areaIds)->get(['id', 'coordinates']);

        foreach ($areas as $area) {
            // if (\App\Helpers\PointInPolygonHelper::check([$lat, $lng], $area->coordinates)) {
                $foundAreaIds[] = $area->id;
            // }
        }
        // if (empty($foundAreaIds)) {
        //     return response()->json(['error' => 'هیچ منطقه فعالی قرار ندارد.'], 404);
        // }

        $areaManagementsForAreas = $areaManagements->whereIn('area_id', $foundAreaIds)->values();
        // if ($areaManagementsForAreas->isEmpty()) {
        //     return response()->json(['error' => 'مدیریت منطقه‌ای برای این مختصات یافت نشد.'], 404);
        // }

        foreach ($areaManagementsForAreas as $am) {
            $shippingMethods = is_array($am->shipping_methods)
                ? $am->shipping_methods
                : json_decode($am->shipping_methods, true);
            if (is_array($shippingMethods)) {
                $deliveryTypeIds = array_merge($deliveryTypeIds, $shippingMethods);
            }
        }
        $deliveryTypeIds = array_merge($deliveryTypeIds, $publicTypeIds);

        $deliveryTypeIds = array_values(array_unique($deliveryTypeIds));

        $deliveryTypes = \App\Models\DeliveryType::whereIn('id', $deliveryTypeIds)->get();

        return response()->json([
            'delivery_methods' => $deliveryTypes
        ]);
    }

    protected function updateOrder($data){
        $order = $this->orderRep->getLatestOrderUser();

        $this->orderRep->update($order->id, $data);

    }
}
