<?php

namespace App\Http\Controllers\V1;

use App\Http\Controllers\Controller;
use App\Models\Branch;
use App\Models\BranchHoliday;
use App\Models\Category;
use App\Models\Product;
use App\Models\Variety;
use App\Repositories\Branch\BranchRepository;
use App\Repositories\BranchHoliday\BranchHolidayRepository;
use App\Repositories\Delivery\DeliveryRepository;
use Carbon\Carbon;


class BranchController extends Controller
{
    protected $branchRep;

    public function __construct()
    {
        $this->branchRep = app(BranchRepository::class);
    }

    public function getActivityTime($branchId)
    {
        $times = $this->branchRep->getActivityTime($branchId);
        return response()->json(['data' => $times]);
    }

    public function show($id)
    {
        $branch = $this->branchRep->showC($id);
        if (!$branch)
            return response()->json(['message' => 'برنچی یافت نشد'], 404);
        return response()->json(['data' => $branch]);
    }

    public function getGallery($branchId)
    {
        $galleries = $this->branchRep->getGallery($branchId);
        return response()->json(['data' => $galleries]);
    }

    public function getNowStatus($branchId)
    {
        $active = $this->branchRep->getNowStatus($branchId);
        return response()->json(['data' => $active]);
    }


    public function perOrderTime($branchId)
    {
        $response = $this->branchRep->perOrderTime($branchId);
        return response()->json(['data' => $response['data'], 'status' => $response['status'], 'message' => $response['message']], $response['status']);
    }


    public function getActivityDay($branchId)
    {
        $active = $this->branchRep->getActivityDay($branchId);
        return response()->json(['data' => $active]);
    }

    public function getAllId()
    {
        return Branch::all()->pluck('id')->toArray() ?? [];
    }


    public function deliveryTypes($branchId)
    {
        $deliveryTypes = app(DeliveryRepository::class)->getDeliveryTypes($branchId);
        return response()->json($deliveryTypes, $deliveryTypes['status']);
    }


    public function sitemapList()
    {
        return response()->json(["status" => 200, "message" => "", "data" => ["branches" => $this->branchRep->sitemapList()]]);
    }

    public function copyBranchData($fromBranchId, $toBranchId)
    {
        $categories = Category::where('branch_id', $fromBranchId)->get();
        foreach ($categories as $category) {
            $newCategory = $category->replicate();
            $newCategory->branch_id = $toBranchId;
            $newCategory->save();

            $products = Product::where('branch_id', $fromBranchId)->where('menu_id', $category->id)->get();
            foreach ($products as $product) {
                $newProduct = $product->replicate();
                $newProduct->branch_id   = $toBranchId;
                $newProduct->menu_id = $newCategory->id;
                $newProduct->save();

                $varieties = Variety::where('product_id', $product->id)->get();

                foreach ($varieties as $variety) {
                    $newVariety = $variety->replicate();
                    $newVariety->product_id = $newProduct->id;
                    $newVariety->save();
                }
            }
        }

        return true;
    }

    /**
     * Get branch status (active/inactive)
     *
     * @param int $branchId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getBranchStatus($branchId)
    {
        $branch = Branch::find($branchId);

        if (!$branch) {
            return response()->json([
                'message' => 'رستوران یافت نشد',
                'status' => 404
            ], 404);
        }

        // Get current activity status (based on time and holidays)
        $currentActivity = $this->branchRep->getNowStatus($branchId);
        $isCurrentlyOpen = $currentActivity !== null;

        // Determine status message
        $statusMessage = '';
        $isActive = false;

        switch ($branch->status) {
            case 'active':
                $isActive = true;
                $statusMessage = 'رستوران فعال است';
                break;
            case 'disable':
                $isActive = false;
                $statusMessage = 'رستوران غیرفعال است';
                break;
            case 'temporarily_disabled':
                $isActive = false;
                $statusMessage = 'رستوران به صورت موقت غیرفعال است';
                break;
            default:
                $isActive = false;
                $statusMessage = 'وضعیت رستوران نامشخص است';
        }

        // Check current holiday status
        $holidayRepository = app(BranchHolidayRepository::class);
        $now = Carbon::now();
        $currentHoliday = $holidayRepository->checkHoliday($now, $now->format('H:i:s'), $branchId);

        $isOnHoliday = $currentHoliday !== null;
        $holidayInfo = null;

        if ($isOnHoliday) {
            $holidayInfo = [
                'id' => $currentHoliday->id,
                'holiday_type' => $currentHoliday->holiday_type,
                'start_date' => $currentHoliday->start_date,
                'end_date' => $currentHoliday->end_date,
                'start_time' => $currentHoliday->start_time ? $currentHoliday->start_time->format('H:i') : null,
                'end_time' => $currentHoliday->end_time ? $currentHoliday->end_time->format('H:i') : null,
                'is_full_day' => $currentHoliday->is_full_day,
                'reason' => $currentHoliday->reason,
                'holiday_reason_type' => $currentHoliday->holiday_reason_type?->value,
                'holiday_reason_type_label' => $currentHoliday->holiday_reason_type?->getLabel(),
            ];
            $statusMessage = 'رستوران در حال حاضر تعطیل است';
            if ($currentHoliday->reason) {
                $statusMessage .= ' - ' . $currentHoliday->reason;
            } elseif ($currentHoliday->holiday_reason_type) {
                $statusMessage .= ' - ' . $currentHoliday->holiday_reason_type->getLabel();
            }
            // When on holiday, branch is not active
            $isActive = false;
        }

        // Get upcoming holidays (next 5)
        $upcomingHolidays = BranchHoliday::active()
            ->forBranch($branchId)
            ->where('start_date', '>=', Carbon::today())
            ->orderBy('start_date')
            ->limit(5)
            ->get()
            ->map(function ($holiday) {
                return [
                    'id' => $holiday->id,
                    'holiday_type' => $holiday->holiday_type,
                    'start_date' => $holiday->start_date,
                    'end_date' => $holiday->end_date,
                    'start_time' => $holiday->start_time ? $holiday->start_time->format('H:i') : null,
                    'end_time' => $holiday->end_time ? $holiday->end_time->format('H:i') : null,
                    'is_full_day' => $holiday->is_full_day,
                    'reason' => $holiday->reason,
                    'holiday_reason_type' => $holiday->holiday_reason_type?->value,
                    'holiday_reason_type_label' => $holiday->holiday_reason_type?->getLabel(),
                ];
            });

        // Get today's activity times
        $todayActivityTimes = $this->branchRep->getActivityDay($branchId);

        return response()->json([
            'data' => [
                'branch_id' => $branch->id,
                'branch_title' => $branch->title,
                'logo' => $branch->logo,
                'status' => $branch->status,
                'is_active' => $isActive,
                'is_currently_open' => $isCurrentlyOpen && !$isOnHoliday,
                'is_on_holiday' => $isOnHoliday,
                'status_message' => $statusMessage,
                'current_activity' => $currentActivity,
                'current_holiday' => $holidayInfo,
                'upcoming_holidays' => $upcomingHolidays,
                'today_activity_times' => $todayActivityTimes,
                'current_time' => $now->format('Y-m-d H:i:s'),
            ],
            'status' => 200,
            'message' => 'عملیات با موفقیت انجام شد'
        ], 200);
    }

}
