<?php

namespace App\Http\Controllers\V1;

use App\Enums\HolidayReasonType;
use App\Http\Controllers\Controller;
use App\Repositories\BranchHoliday\BranchHolidayRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
class BranchHolidayController extends Controller
{
    protected $holidayRepository;

    public function __construct(BranchHolidayRepository $holidayRepository)
    {
        $this->holidayRepository = $holidayRepository;
    }

    /**
     * get holidays
     */
    public function index(Request $request)
    {
        try {

            $filters = $request->only(['status', 'holiday_type', 'holiday_reason_type', 'start_date', 'end_date']);
            $filters['branch_id'] = $request->branch_id;


            $holidays = $this->holidayRepository->getBranchHolidays($filters, $request->branch_id);


            return response()->json([
                'status' => 200,
                'message' => 'تعطیلی‌ها با موفقیت دریافت شدند',
                'data' => $holidays,
                'debug' => [
                    'branch_id' => $request->branch_id,
                    'filters' => $filters,
                    'count' => $holidays->count()
                ]
            ]);
        } catch (\Exception $e) {
            Log::error("Error in index method: " . $e->getMessage());
            Log::error("Stack trace: " . $e->getTraceAsString());

            return response()->json([
                'status' => 500,
                'message' => 'خطا در دریافت تعطیلی‌ها: ' . $e->getMessage(),
                'debug' => [
                    'branch_id' => $request->branch_id ?? 'not set',
                    'error' => $e->getMessage()
                ]
            ], 500);
        }
    }

    /**
     * create holiday
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'holiday_type' => 'required|in:single_date,date_range,weekly_pattern',
                'start_date' => 'required|date',
                'end_date' => 'nullable|date|after_or_equal:start_date',
                'start_time' => 'nullable|date_format:H:i',
                'end_time' => 'nullable|date_format:H:i',
                'is_full_day' => 'nullable|boolean',
                'week_days' => 'nullable|array',
                'week_days.*' => 'string|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
                'repeat_yearly' => 'nullable|boolean',
                'reason' => 'nullable|string|max:255',
                'holiday_reason_type' => 'nullable|in:' . implode(',', array_column(HolidayReasonType::cases(), 'value')),
                'status' => 'nullable|in:active,inactive'
            ]);
            Log::info($validator->errors());
            if ($validator->fails()) {
                return response()->json([
                    'status' => 422,
                    'message' => 'خطا در اعتبارسنجی داده‌ها',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->all();
            $data['branch_id'] = $request->branch_id;

            // تنظیم مقادیر پیش‌فرض
            $data['is_full_day'] = $request->get('is_full_day', true);
            $data['repeat_yearly'] = $request->get('repeat_yearly', false);
            $data['status'] = $request->get('status', 'active');

            $holiday = $this->holidayRepository->createHoliday($data);

            return response()->json([
                'status' => 201,
                'message' => 'تعطیلی با موفقیت ایجاد شد',
                'data' => $holiday
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'خطا در ایجاد تعطیلی: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * show holiday
     */
    public function show(Request $request, $holidayId)
    {
        try {
            $holiday = $this->holidayRepository->show($holidayId);

            if (!$holiday || $holiday->branch_id != $request->branch_id) {
                return response()->json([
                    'status' => 404,
                    'message' => 'تعطیلی یافت نشد'
                ], 404);
            }

            return response()->json([
                'status' => 200,
                'message' => 'تعطیلی با موفقیت دریافت شد',
                'data' => $holiday
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'خطا در دریافت تعطیلی: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * update holiday
     */
    public function update(Request $request, $holidayId)
    {
        try {
            $validator = Validator::make($request->all(), [
                'holiday_type' => 'nullable|in:single_date,date_range,weekly_pattern',
                'start_date' => 'nullable|date',
                'end_date' => 'nullable|date|after_or_equal:start_date',
                'start_time' => 'nullable|date_format:H:i',
                'end_time' => 'nullable|date_format:H:i',
                'is_full_day' => 'nullable|boolean',
                'week_days' => 'nullable|array',
                'week_days.*' => 'string|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
                'repeat_yearly' => 'nullable|boolean',
                'reason' => 'nullable|string|max:255',
                'holiday_reason_type' => 'nullable|in:' . implode(',', array_column(HolidayReasonType::cases(), 'value')),
                'status' => 'nullable|in:active,inactive'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 422,
                    'message' => 'خطا در اعتبارسنجی داده‌ها',
                    'errors' => $validator->errors()
                ], 422);
            }

            $holiday = $this->holidayRepository->show($holidayId);

            if (!$holiday || $holiday->branch_id != $request->branch_id) {
                return response()->json([
                    'status' => 404,
                    'message' => 'تعطیلی یافت نشد'
                ], 404);
            }

            $updatedHoliday = $this->holidayRepository->updateHoliday($holidayId, $request->all());

            return response()->json([
                'status' => 200,
                'message' => 'تعطیلی با موفقیت به‌روزرسانی شد',
                'data' => $updatedHoliday
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'خطا در به‌روزرسانی تعطیلی: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * delete holiday
     */
    public function destroy(Request $request, $holidayId)
    {
        try {
            $holiday = $this->holidayRepository->show($holidayId);

            if (!$holiday || $holiday->branch_id != $request->branch_id) {
                return response()->json([
                    'status' => 404,
                    'message' => 'تعطیلی یافت نشد'
                ], 404);
            }

            $this->holidayRepository->deleteHoliday($holidayId);

            return response()->json([
                'status' => 200,
                'message' => 'تعطیلی با موفقیت حذف شد'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'خطا در حذف تعطیلی: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * check branch status
     */
    public function checkBranchStatus(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'date' => 'nullable|date',
                'time' => 'nullable|date_format:H:i:s'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 422,
                    'message' => 'خطا در اعتبارسنجی داده‌ها',
                    'errors' => $validator->errors()
                ], 422);
            }

            $date = $request->get('date');
            $time = $request->get('time');

            $isOpen = $this->holidayRepository->isBranchOpen($request->branch_id, $date, $time);

            return response()->json([
                'status' => 200,
                'message' => 'وضعیت شعبه بررسی شد',
                'data' => [
                    'branch_id' => $request->branch_id,
                    'is_open' => $isOpen,
                    'check_date' => $date ?: Carbon::now()->format('Y-m-d'),
                    'check_time' => $time ?: Carbon::now()->format('H:i:s')
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'خطا در بررسی وضعیت شعبه: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * get upcoming holidays
     */
    public function getUpcomingHolidays(Request $request)
    {
        try {
            $limit = $request->get('limit', 10);
            $holidays = $this->holidayRepository->getUpcomingHolidays($limit, $request->branch_id);

            return response()->json([
                'status' => 200,
                'message' => 'تعطیلی‌های آینده با موفقیت دریافت شدند',
                'data' => $holidays
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'خطا در دریافت تعطیلی‌های آینده: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * get holidays in range
     */
    public function getHolidaysInRange(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'start_date' => 'required|date',
                'end_date' => 'required|date|after_or_equal:start_date'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 422,
                    'message' => 'خطا در اعتبارسنجی داده‌ها',
                    'errors' => $validator->errors()
                ], 422);
            }

            $startDate = $request->get('start_date');
            $endDate = $request->get('end_date');

            $holidays = $this->holidayRepository->getActiveHolidaysInRange($request->branch_id, $startDate, $endDate);

            return response()->json([
                'status' => 200,
                'message' => 'تعطیلی‌های بازه زمانی با موفقیت دریافت شدند',
                'data' => $holidays
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'خطا در دریافت تعطیلی‌های بازه زمانی: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * get holiday reason types
     */
    public function getHolidayReasonTypes()
    {
        try {
            $reasonTypes = HolidayReasonType::getOptions();

            return response()->json([
                'status' => 200,
                'message' => 'انواع علت تعطیلی با موفقیت دریافت شدند',
                'data' => $reasonTypes
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'خطا در دریافت انواع علت تعطیلی: ' . $e->getMessage()
            ], 500);
        }
    }
}
