<?php

namespace App\Http\Controllers\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\CustomerRequest;
use App\Services\CustomerService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class CustomerBranchController extends Controller
{
    protected $customerService;

    public function __construct(CustomerService $customerService)
    {
        $this->customerService = $customerService;
    }

    /**
     * Get all customers for a specific branch
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $branchId = $request->input('branch_id');

            if (!$branchId) {
                return response()->json([
                    'status' => false,
                    'message' => 'Branch ID is required'
                ], 400);
            }

            $customers = $this->customerService->getCustomersForBranch(
                $branchId,
                $request->input('search', ''),
                $request->input('per_page', 20)
            );

            return response()->json([
                'status' => true,
                'data' => $customers
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error fetching customers: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a new customer for a branch
     */
    public function store(CustomerRequest $request): JsonResponse
    {
        try {
            $customer = $this->customerService->createCustomer($request->validated());

            return response()->json([
                'status' => true,
                'message' => 'Customer created successfully',
                'data' => $customer
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error creating customer: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show a specific customer
     */
    public function show($id): JsonResponse
    {
        try {
            $customer = $this->customerService->getCustomerById($id);

            return response()->json([
                'status' => true,
                'data' => $customer
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Customer not found'
            ], 404);
        }
    }

    /**
     * Update a customer
     */
    public function update(CustomerRequest $request, $id): JsonResponse
    {
        try {
            $customer = $this->customerService->updateCustomer($id, $request->validated());

            return response()->json([
                'status' => true,
                'message' => 'Customer updated successfully',
                'data' => $customer
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error updating customer: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a customer
     */
    public function destroy($id): JsonResponse
    {
        try {
            $this->customerService->deleteCustomer($id);

            return response()->json([
                'status' => true,
                'message' => 'Customer deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error deleting customer: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search customers by name, family, or mobile
     */
    public function search(Request $request): JsonResponse
    {
        try {
            $branchId = $request->input('branch_id');
            $search = $request->input('search');

            if (!$branchId) {
                return response()->json([
                    'status' => false,
                    'message' => 'Branch ID is required'
                ], 400);
            }

            $customers = $this->customerService->searchCustomers($branchId, $search);

            return response()->json([
                'status' => true,
                'data' => $customers
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error searching customers: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get customer statistics for a branch
     */
    public function stats(Request $request): JsonResponse
    {
        try {
            $branchId = $request->input('branch_id');

            if (!$branchId) {
                return response()->json([
                    'status' => false,
                    'message' => 'Branch ID is required'
                ], 400);
            }

            $stats = $this->customerService->getCustomerStats($branchId);

            return response()->json([
                'status' => true,
                'data' => $stats
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error fetching customer statistics: ' . $e->getMessage()
            ], 500);
        }
    }
}
