<?php

namespace App\Http\Controllers\V1;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Repositories\Address\AddressRepository;
use App\Repositories\Delivery\DeliveryRepository;
use App\Repositories\DeliveryType\DeliveryTypeRepository;
use App\Repositories\Order\OrderRepository;
use App\Repositories\User\UserRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use App\Services\AreaManagementService;

class OrderController extends Controller
{
    protected $orderRep;
    protected $addressRep;

    public function __construct()
    {
        $this->orderRep = app(OrderRepository::class);
        $this->addressRep = app(AddressRepository::class);
    }

    public function getAllWithItems(Request $request)
    {
        $now = \Carbon\Carbon::now();

        $token = null;
        if (isset($request->guest_token))
            $token = $request->guest_token;

        if($request->address_id && $request->branch_id){
            $result = app(AreaManagementService::class)->getActiveAreaForAddress(
                    $request->address_id,
                    $request->branch_id,
                    $now->format('l'),
                    $now->format('H:i:s'),
                    $request->lat ?? null,
                    $request->lng ?? null
                    );
        }else{
            $result = null;
        }

        $order = $this->orderRep->getLatestOrderWithItems($token, $request->delivery_type_id ?? null, $result ?? null, $request->address_id ?? null);
        if ($order['order'] instanceof Order)
            return response()->json(['data' => $order]);
        return response()->json(['data' => null], 404);
    }


    public function addProductCart(Request $request)
    {
        $deliveryType = null;
        if ($request->has('delivery_type_id')) {
            $deliveryType = \App\Models\DeliveryType::find($request->delivery_type_id);
        }
        $rules = [
            'variety_id' => 'required|numeric|exists:varieties,id',
            'count' => 'required|numeric',
            'extraIds' => 'nullable|string',
            'time' => 'nullable|date_format:H:i:s',
            'week_day' => 'nullable|string|in:sunday,monday,tuesday,wednesday,thursday,friday,saturday',
        ];
        // if (!$deliveryType || !in_array($deliveryType->delivery_method, ['serve', 'actual'])) {
        //     $rules['address_id'] = 'required|numeric|exists:addresses,id';
        // }
        // $validator = Validator::make($request->all(), $rules, [
        //     'address_id.required' => 'ابتدا برای ثبت سفارش به حداقل یک آدرس در منطقه این رستوران نیاز دارید'
        // ]);
        $user_loged = auth()->user() ? true : false;
        // if ($validator->fails()) {
        //     return response()->json(['message' => $validator->errors()->first(), 'status' => 400, 'data' => [], "user_loged"=>$user_loged], 400);
        // }
        $order = $this->orderRep->addCartOrder($request->all());
        return response()->json(['data' => $order['data'], 'message' => $order['message'], 'status' => $order['status'], "user_loged"=>$user_loged], $order['status']);
    }

    public function storeOrUpdateOrderWithList(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'carts' => 'required|array',
            'carts.*.variety_id' => 'required|exists:varieties,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->first()], 400);
        }
        $this->orderRep->OrderListDeleteCart();
        foreach ($request->carts as $cart) {
            $order = $this->orderRep->addCartOrder($cart);
        }
        return response()->json(['data' => $order['data'], 'message' => $order['message'], 'status' => $order['status']], $order['status']);
    }

    public function minesProductCart(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'variety_id' => 'required|numeric|exists:varieties,id',
            'count' => 'required|numeric',
            'extraIds' => 'nullable|string',
        ]);
        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->first(), 'status' => 400, 'data' => []], 400);
        }
        $order = $this->orderRep->minesCartOrder($request->all());
        return response()->json(['data' => $order['data'], 'message' => $order['message'], 'status' => $order['status']]);
    }

    public function getCountItem()
    {
        $orderCount = $this->orderRep->getCountItem();
        return response()->json(['data' => $orderCount]);
    }

    public function deleteProductCart(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'variety_id' => 'required|numeric|exists:varieties,id',
            'count' => 'required|numeric',
            'extraIds' => 'nullable|string',
        ]);
        if ($validator->fails())
            return response()->json(['message' => $validator->errors()->first()], 400);
        $order = $this->orderRep->deleteCartOrder($request->all());
        if ($order['order'] instanceof Order)
            return response()->json(['data' => $order]);
        return response()->json(['data' => null], 404);
    }

    public function setDeliveryInfo(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'address_id' => 'nullable|numeric|exists:addresses,id',
        ]);
        if ($validator->fails())
            return response()->json(['message' => $validator->errors()->first()], 400);

        $order = $this->orderRep->getLatestOrderUser();

        if ($order->delivery_type_id == null)
            return response()->json(['status' => 400, 'message' => 'لطفا نحوه دریافت رو انتخاب کنید', 'data' => null], 400);

        if ($order->branch_id == null || $order->orderCart == null)
            return response()->json(['status' => 400, 'message' => 'اطلاعات نادرست است ', 'data' => null], 400);

        $check = app(DeliveryRepository::class)->checkAccessAddress($order->id, $order->delivery_type_id, $request->address_id);
        Log::info($check);
        if ($check['data']['check'] == false)
            return response()->json(['status' => 400, 'message' => $check['message'], 'data' => null], 400);

        $this->orderRep->update($order->id, ['address_id' => $request->address_id, 'delivery_type_id' => $order->delivery_type_id]);
        return response()->json(['status' => 200, 'message' => $check['message'] ?? 'آدرس با موفقیت ثبت شد', 'data' => $this->orderRep->getLatestOrderWithItems()], 200);
    }

    public function changeDeliveryType(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'delivery_type_id' => 'required|numeric|exists:delivery_types,id',
            'address_id' => 'nullable|numeric|exists:addresses,id',
        ]);
        if ($validator->fails())
            return response()->json(['message' => $validator->errors()->first()], 400);

        $order = $this->orderRep->getLatestOrderUser();

        $deliveryType = app(DeliveryTypeRepository::class)->show($request->delivery_type_id);

        if ($order->branch_id == null || $order->orderCart == null || $deliveryType == null) {
            Log::info($order);
            Log::info($order->branch_id);
            Log::info($order->orderCart);
            Log::info($deliveryType);
            return response()->json(['status' => 400, 'message' => 'اطلاعات نادرست است ', 'data' => null], 400);
        }
        if ($deliveryType->delivery_method !== 'delivery') {

            $price = $order->order_price - $order->delivery_price;
            $this->orderRep->update($order->id, ['delivery_price' => 0, 'order_price' => $price, 'delivery_type_id' => $request->delivery_type_id, 'delivery_type' => json_encode($deliveryType), 'address_id' => $request->address_id]);

        } else {
            $this->orderRep->update($order->id, ['delivery_type_id' => $request->delivery_type_id, 'delivery_type' => json_encode($deliveryType), 'address_id' => $request->address_id]);

        }

        return response()->json(['status' => 200, 'message' => $check['message'] ?? 'آدرس با موفقیت ثبت شد', 'data' => $this->orderRep->getLatestOrderWithItems()], 200);
    }

    public function checkTimeSet(Request $request)
    {
//        format date 1400/05/03 19:00:00
        $check = $this->addressRep->checkAddress($request->date, $request->branchId);
        return response()->json(['data' => $check['data'] ?? null, 'message' => $check['message']], $check['status']);
    }

    public function getListAddressAccessCart(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'address_title' => 'nullable|string',
            'address_area' => 'nullable|numeric|exists:areas,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->first(), 'status' => 400, 'data' => []], 400);
        }
        $address = $this->orderRep->getListAddressAccessCart($request);
        return response()->json(['data' => $address]);
    }

    public function orderListBefore()
    {
        $orders = $this->orderRep->orderListBefore();
        return response()->json(['data' => $orders]);
    }

    public function getListGateways(Request $request)
    {
        $data = $this->orderRep->getListGateways();
        return response()->json(['data' => $data]);
    }

    public function checkCoupon(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'coupon' => 'required'
        ]);
        if ($validator->fails())
            return response()->json(['message' => $validator->errors()->first()], 400);
        $order = $this->orderRep->checkCoupon($request->all());
        return response()->json($order, $order['status']);
    }

    public function getOrderUser(Request $request)
    {
        $orders = app(UserRepository::class)->getOrders($request->all());
        if (!$orders) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 500);
        }
        return response()->json(['data' => $orders]);
    }

    public function getOrder(Request $request, $id)
    {
        $order = $this->orderRep->showWithUser($id);
        if (!$order) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 500);
        }

        // محاسبه قیمت‌ها مشابه با getAllWithItems
        $price = $this->orderRep->sumPriceOrder($order);

        return response()->json(['data' => [
            'order' => $order,
            'price' => $price
        ]]);
    }

    public function createComment(Request $request, $id)
    {
        $orders = $this->orderRep->createComment($request->all(), $id);
        return response()->json($orders, $orders['status']);
    }

    /**
     * حذف سبد خرید فعال کاربر بر اساس user_id
     */
    public function deleteUserActiveCart(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|numeric|exists:users,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => $validator->errors()->first(),
                'status' => 400,
                'data' => null
            ], 400);
        }

        $userId = $request->user_id;
        $result = $this->orderRep->deleteUserActiveOrder($userId);

        return response()->json([
            'message' => $result['message'],
            'status' => $result['status'],
            'data' => $result['data']
        ], $result['status']);
    }


}
