<?php

namespace App\Http\Controllers\V1;

use App\Http\Controllers\Controller;
use App\Repositories\PopupAlarm\RepositoryInterface as PopupAlarmRepository;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Morilog\Jalali\Jalalian;
use App\Enums\PopupAlarmPageType;

class PopupAlarmController extends Controller
{
    protected $popupAlarmRepository;

    public function __construct(PopupAlarmRepository $popupAlarmRepository)
    {
        $this->popupAlarmRepository = $popupAlarmRepository;
    }

    /**
     * ایجاد آلارم جدید برای رستوران
     */
    public function store(Request $request): JsonResponse
    {
        $user = Auth::user();

        if ($user->role_id != 1 && $user->role_id != 3) {
            return response()->json([
                'success' => false,
                'message' => 'شما رستوران فعالی ندارید'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'branch_id' => 'nullable|exists:branches,id',
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'image' => 'nullable|url',
            'start_time' => 'required|date',
            'end_time' => 'nullable|date|after:start_time',
            'page_type' => 'required|in:home,cart_check,cart_payment',
        ]);
        Log::info($validator->errors());
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی داده‌ها',
                'errors' => $validator->errors()
            ], 422);
        }

        $branchId = $request->branch_id ?? $user->target_role_id;

        $data = $request->only(['branch_id','image', 'title', 'message', 'start_time', 'end_time', 'page_type']);
        $data['branch_id'] = $branchId;
        $data['status'] = 'active';
        $alarm = $this->popupAlarmRepository->create($data);

        return response()->json([
            'success' => true,
            'message' => 'آلارم با موفقیت ایجاد شد',
            'data' => [
                'id' => $alarm->id,
                'title' => $alarm->title,
                'message' => $alarm->message,
                'image' => $alarm->image,
                'start_time' => $alarm->start_time_jalali,
                'end_time' => $alarm->end_time_jalali,
                'status' => $alarm->status,
                'page_type' => $alarm->page_type->value,
                'page_type_label' => $alarm->page_type_label,
                'created_at' => Jalalian::fromCarbon($alarm->created_at)->format('Y/m/d H:i')
            ]
        ], 201);
    }

    /**
     * دریافت آلارم‌های فعال که کاربر ندیده
     */
    public function getUnseenAlarms(Request $request): JsonResponse
    {
        $branchId = $request->get('branch_id');
        $userId   = Auth::guard('api')->id();  
        $alarms = $this->popupAlarmRepository->getUnseenActiveAlarmsForUser($userId??null, $branchId);

        $alarmsData = $alarms->map(function ($alarm) {
            return [
                'id' => $alarm->id,
                'branch_id' => $alarm->branch_id,
                'branch_title' => $alarm->branch->title ?? '',
                'title' => $alarm->title,
                'message' => $alarm->message,
                'image' => $alarm->image ? url($alarm->image) : null,
                'start_time' => $alarm->start_time_jalali,
                'end_time' => $alarm->end_time_jalali,
                'page_type' => $alarm->page_type->value,
                'page_type_label' => $alarm->page_type_label,
                'created_at' => Jalalian::fromCarbon($alarm->created_at)->format('Y/m/d H:i')
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $alarmsData
        ]);
    }

    /**
     * ثبت مشاهده آلارم توسط کاربر
     */
    public function markAsViewed(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'alarm_id' => 'required|exists:popup_alarms,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی داده‌ها',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = Auth::user();
        $alarmId = $request->alarm_id;

        // بررسی اینکه کاربر قبلاً این آلارم را دیده یا نه
        if ($this->popupAlarmRepository->hasUserSeenAlarm($alarmId, $user->id)) {
            return response()->json([
                'success' => false,
                'message' => 'شما قبلاً این آلارم را مشاهده کرده‌اید'
            ], 400);
        }

        $view = $this->popupAlarmRepository->markAsViewed($alarmId, $user->id);

        return response()->json([
            'success' => true,
            'message' => 'آلارم به عنوان مشاهده شده ثبت شد',
            'data' => [
                'viewed_at' => $view->viewed_at_jalali
            ]
        ]);
    }

    /**
     * دریافت لیست آلارم‌های رستوران
     */
    public function getBranchAlarms(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
            'per_page' => 'nullable|integer|min:1|max:100'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی داده‌ها',
                'errors' => $validator->errors()
            ], 422);
        }

        $branchId = $request->branch_id;
        $perPage = $request->get('per_page', 15);

        $alarms = $this->popupAlarmRepository->getBranchAlarms($branchId, $perPage);

        $alarmsData = $alarms->map(function ($alarm) {
            return [
                'id' => $alarm->id,
                'title' => $alarm->title,
                'message' => $alarm->message,
                'image' => $alarm->image ? url($alarm->image) : null,
                'start_time' => $alarm->start_time_jalali,
                'end_time' => $alarm->end_time_jalali,
                'status' => $alarm->status,
                'page_type' => $alarm->page_type->value,
                'page_type_label' => $alarm->page_type_label,
                'is_active' => $alarm->isActive(),
                'total_views' => $alarm->views->count(),
                'created_at' => Jalalian::fromCarbon($alarm->created_at)->format('Y/m/d H:i')
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $alarmsData,
            'pagination' => [
                'current_page' => $alarms->currentPage(),
                'last_page' => $alarms->lastPage(),
                'per_page' => $alarms->perPage(),
                'total' => $alarms->total()
            ]
        ]);
    }

    /**
     * به‌روزرسانی آلارم
     */
    public function update(Request $request, int $id): JsonResponse
    {
        $user = Auth::user();

        // بررسی دسترسی به آلارم
        $alarm = \App\Models\PopupAlarm::findOrFail($id);

        // اگر کاربر ادمین نیست، فقط می‌تواند آلارم‌های رستوران خود را ویرایش کند
        if ($user->role_id != 3) {
            return response()->json([
                'success' => false,
                'message' => 'شما دسترسی به این آلارم ندارید'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'image' => 'nullable|url',
            'start_time' => 'required|date',
            'end_time' => 'nullable|date|after:start_time',
            'status' => 'required|in:active,inactive',
            'page_type' => 'required|in:home,cart_check,cart_payment'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی داده‌ها',
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $request->only(['title', 'message','image', 'start_time', 'end_time', 'status', 'page_type']);

        $updated = $this->popupAlarmRepository->update($id, $data);

        if ($updated) {
            return response()->json([
                'success' => true,
                'message' => 'آلارم با موفقیت به‌روزرسانی شد'
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'خطا در به‌روزرسانی آلارم'
        ], 500);
    }

    /**
     * حذف آلارم
     */
    public function destroy(int $id): JsonResponse
    {
        $user = Auth::user();

        $alarm = \App\Models\PopupAlarm::findOrFail($id);

        if ($user->role_id != 3) {
            return response()->json([
                'success' => false,
                'message' => 'شما دسترسی به این آلارم ندارید'
            ], 403);
        }

        $deleted = $this->popupAlarmRepository->delete($id);

        if ($deleted) {
            return response()->json([
                'success' => true,
                'message' => 'آلارم با موفقیت حذف شد'
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'خطا در حذف آلارم'
        ], 500);
    }

    /**
     * دریافت آمار مشاهده آلارم
     */
    public function getViewStats(int $id): JsonResponse
    {
        $user = Auth::user();

        $alarm = \App\Models\PopupAlarm::findOrFail($id);

        if ($user->role_id != 1) {
            if (!$user->vendor || !$user->vendor->branches()->where('id', $alarm->branch_id)->exists()) {
                return response()->json([
                    'success' => false,
                    'message' => 'شما دسترسی به این آلارم ندارید'
                ], 403);
            }
        }

        $stats = $this->popupAlarmRepository->getAlarmViewStats($id);

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }

    /**
     * دریافت لیست آلارم‌های رستوران خود (برای رستوران‌ها)
     */
    public function getMyBranchAlarms(Request $request): JsonResponse
    {
        $user = Auth::user();

        if ($user->role_id != 3) {
            return response()->json([
                'success' => false,
                'message' => 'شما رستوران فعالی ندارید'
            ], 403);
        }

        $branchId = $user->target_role_id;
        $perPage = $request->get('per_page', 15);
        $alarms = $this->popupAlarmRepository->getBranchAlarms($branchId, $perPage);
        $alarmsData = $alarms->map(function ($alarm) {
            return [
                'id' => $alarm->id,
                'title' => $alarm->title,
                'message' => $alarm->message,
                'image' => $alarm->image ? url($alarm->image) : null,
                'start_time' => $alarm->start_time_jalali,
                'end_time' => $alarm->end_time_jalali,
                'status' => $alarm->status,
                'page_type' => $alarm->page_type->value,
                'page_type_label' => $alarm->page_type_label,
                'is_active' => $alarm->isActive(),
                'total_views' => $alarm->views->count(),
                'created_at' => Jalalian::fromCarbon($alarm->created_at)->format('Y/m/d H:i')
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $alarmsData,
            'pagination' => [
                'current_page' => $alarms->currentPage(),
                'last_page' => $alarms->lastPage(),
                'per_page' => $alarms->perPage(),
                'total' => $alarms->total()
            ]
        ]);
    }

    public function getAlarmById(Request $request): JsonResponse
    {
        $user = Auth::user();
        $id = $request->id;
        $alarm = $this->popupAlarmRepository->getAlarmById($id);
        if ($user->role_id != 3) {
            return response()->json([
                'success' => false,
                'message' => 'شما دسترسی به این آلارم ندارید'
            ], 403);

        }
        return response()->json([
            'success' => true,
            'data' => $alarm
        ]);
    }

    /**
     * دریافت لیست انواع صفحات نمایش
     */
    public function getPageTypes(): JsonResponse
    {
        return response()->json([
            'success' => true,
            'data' => PopupAlarmPageType::getAll()
        ]);
    }
}
