<?php

namespace App\Http\Controllers\V1;

use App\Enums\ProductStatus;
use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Product;
use App\Repositories\Category\CategoryRepository;
use App\Repositories\Product\ProductRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ProductController extends Controller
{

    protected $productRepository;
    protected $categoryRepository;

    public function __construct()
    {
        $this->productRepository = app(ProductRepository::class);
        $this->categoryRepository = app(CategoryRepository::class);
    }

    public function show($id)
    {
        $product = $this->productRepository->getProductWithAllRel($id);
        return response()->json(['data' => $product]);
    }

    public function getCategoryAll(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'nullable|numeric|exists:branches,id',
            'title' => 'nullable',
            'time' => 'nullable|date_format:H:i:s',
        ]);
        if ($validator->fails())
            return response()->json([
                'message' => $validator->errors()->first(),
                'data' => [],
                'status' => 400,
            ], 400);

        $data = $request->all();
        $data['select'] = ['id', 'title', 'img', 'parent_id', 'branch_id', 'start_serving', 'end_serving'];
        $category = $this->categoryRepository->getCategoryAllWithProduct($data);
        return response()->json([
            'message' => '',
            'data' => $category,
            'status' => 200,
        ]);
    }

    public function index(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'nullable|numeric|exists:branches,id',
            'category_id' => 'nullable|numeric|exists:categories,id',
            'materials_id' => 'nullable',
            'price_start' => 'nullable',
            'price_end' => 'nullable',
            'order_by' => 'nullable',
            'paginate' => 'nullable',
//            'has_discount' => 'nullable|boolean',
            'sort_rate' => 'nullable|boolean',
            'lat' => 'nullable|numeric|between:-90,90',
            'lng' => 'nullable|numeric|between:-180,180',
        ]);
        if ($validator->fails())
            return response()->json(['error' => $validator->errors()->first()], 400);

        return response()->json(['data' => app(ProductRepository::class)->getByFilterFront($request->all())], 200);
    }

    public function getItems(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'nullable',
            'category_id' => 'nullable|numeric|exists:categories,id',
            'vendor_id' => 'nullable|numeric|exists:vendors,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $vendors = Item::
        where(function ($query) use ($request) {
            if ($request->title != null and $request->title != " ") {
                $query->where('title', 'like', '%' . $request->title . '%');
            }
        })->where(function ($query) use ($request) {
            if ($request->area_id != null) {
                $query->whereHas('area', function ($query) use ($request) {
                    $query->where('id', $request->area_id);
                });
            }
        })->where(function ($query) use ($request) {
            if ($request->category_id != null) {
                $query->whereHas('category', function ($query) use ($request) {
                    $query->where('id', $request->category_id);
                });
            }
        })->where(function ($query) use ($request) {
            if ($request->lat != null and $request->lng != null) {
                $lat = $request->lat;
                $lng = $request->lng;
                $radius = 10;
                $haversine = "(6371 * acos(cos(radians($lat)) * cos(radians(lat)) * cos(radians(lng) - radians($lng)) + sin(radians($lat)) * sin(radians(lat))))";
                $query->selectRaw("{$haversine} AS distance")
                    ->whereRaw("{$haversine} < ?", [$radius]);
            }
        })->paginate(10); // 10 vendors per page, adjust as needed
        return response()->json(['data' => $vendors], 200);
    }

    public function getCalories($id)
    {
        return response()->json($this->productRepository->getAllCalories($id));
    }

    public function sitemapListProducts()
    {
        return response()->json(["status" => 200, "message" => "", "data" => ["products" => $this->productRepository->sitemapListProducts()]]);
    }
}
