<?php

namespace App\Http\Controllers\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\UserOrderRequest;
use App\Http\Resources\UserOrderResource;
use App\Http\Resources\UserOrderStatsResource;
use App\Services\UserOrderService;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class UserOrderController extends Controller
{
    protected UserOrderService $userOrderService;

    public function __construct(UserOrderService $userOrderService)
    {
        $this->userOrderService = $userOrderService;
    }

    /**
     * Get user's order history from a specific branch.
     */
    public function getUserBranchOrders(UserOrderRequest $request): JsonResponse
    {
        try {
            $validated = $request->validated();
            $userId = $validated['user_id'];
            $branchId = $validated['branch_id'];
            $perPage = $validated['per_page'] ?? 15;

            // Get filters
            $filters = $request->only([
                'order_status',
                'payment_status',
                'start_date',
                'end_date',
                'min_amount',
                'max_amount'
            ]);

            // Get orders
            $orders = $this->userOrderService->getUserBranchOrders($userId, $branchId, $filters, $perPage);

            // Get user and branch info
            $info = $this->userOrderService->getUserBranchInfo($userId, $branchId);

            return response()->json([
                'status' => 'success',
                'message' => 'تاریخچه خرید کاربر از فروشگاه با موفقیت دریافت شد',
                'data' => [
                    'user' => $info['user'],
                    'branch' => $info['branch'],
                    'orders' => $orders
                ]
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در اعتبارسنجی داده‌ها',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('خطا در دریافت تاریخچه خرید کاربر: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage(),
                'error' => config('app.debug') ? $e->getMessage() : 'خطا در دریافت تاریخچه خرید کاربر'
            ], 500);
        }
    }

    /**
     * Get user's order statistics from a specific branch.
     */
    public function getUserBranchOrderStats(UserOrderRequest $request): JsonResponse
    {
        try {
            $validated = $request->validated();
            $userId = $validated['user_id'];
            $branchId = $validated['branch_id'];

            // Get filters
            $filters = $request->only([
                'order_status',
                'payment_status',
                'start_date',
                'end_date',
                'min_amount',
                'max_amount'
            ]);

            // Get statistics
            $stats = $this->userOrderService->getUserBranchOrderStats($userId, $branchId, $filters);

            // Get user and branch info
            $info = $this->userOrderService->getUserBranchInfo($userId, $branchId);

            return response()->json([
                'status' => 'success',
                'message' => 'آمار خرید کاربر از فروشگاه با موفقیت دریافت شد',
                'data' => [
                    'user' => $info['user'],
                    'branch' => $info['branch'],
                    'statistics' => new UserOrderStatsResource($stats)
                ]
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در اعتبارسنجی داده‌ها',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('خطا در دریافت آمار خرید کاربر: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage(),
                'error' => config('app.debug') ? $e->getMessage() : 'خطا در دریافت آمار خرید کاربر'
            ], 500);
        }
    }

    /**
     * Get user's favorite products from a specific branch.
     */
    public function getUserFavoriteProducts(UserOrderRequest $request): JsonResponse
    {
        try {
            $validated = $request->validated();
            $userId = $validated['user_id'];
            $branchId = $validated['branch_id'];
            $limit = $request->get('limit', 10);

            // Get favorite products
            $favoriteProducts = $this->userOrderService->getUserFavoriteProducts($userId, $branchId, $limit);

            // Get user and branch info
            $info = $this->userOrderService->getUserBranchInfo($userId, $branchId);

            return response()->json([
                'status' => 'success',
                'message' => 'محصولات محبوب کاربر از فروشگاه با موفقیت دریافت شد',
                'data' => [
                    'user' => $info['user'],
                    'branch' => $info['branch'],
                    'favorite_products' => $favoriteProducts
                ]
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در اعتبارسنجی داده‌ها',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('خطا در دریافت محصولات محبوب کاربر: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage(),
                'error' => config('app.debug') ? $e->getMessage() : 'خطا در دریافت محصولات محبوب کاربر'
            ], 500);
        }
    }
}
