<?php

namespace App\Http\Controllers\V1;

use App\Http\Controllers\Controller;
use App\Models\Branch;
use App\Models\Category;
use App\Repositories\Category\CategoryRepository;
use App\Repositories\Vendor\VendorRepository;
use Illuminate\Http\Request;
use App\Models\Vendor;
use Illuminate\Support\Facades\Validator;

class VendorController extends Controller
{


    protected $vendorRep;

    public function __construct()
    {
        $this->vendorRep = app(VendorRepository::class);
    }

    public function index()
    {
        $vendors = Vendor::all();
        return response()->json($vendors);
    }

    public function show($id)
    {
        $vendor = Vendor::find($id);

        if (!$vendor) {
            return response()->json(['message' => 'Vendor not found'], 404);
        }

        return response()->json($vendor);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required',
            'subdomain' => 'required',
            // Add validation rules for other fields
        ]);

        $vendor = Vendor::create($data);

        return response()->json($vendor, 201);
    }

    public function update(Request $request, $id)
    {
        $vendor = Vendor::find($id);

        if (!$vendor) {
            return response()->json(['message' => 'Vendor not found'], 404);
        }

        $data = $request->validate([
            'name' => 'required',
            'subdomain' => 'required',
            // Add validation rules for other fields
        ]);

        $vendor->update($data);

        return response()->json($vendor);
    }

    public function destroy($id)
    {
        $vendor = Vendor::find($id);

        if (!$vendor) {
            return response()->json(['message' => 'Vendor not found'], 404);
        }

        $vendor->delete();

        return response()->json(['message' => 'Vendor deleted']);
    }

    public function getAllBranches(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'nullable',
            'area_id' => 'nullable|numeric|exists:areas,id',
            'category_id' => 'nullable|numeric|exists:categories,id',
            'lat' => 'nullable|numeric|between:-90,90',
            'lng' => 'nullable|numeric|between:-180,180',
            "order_by" => 'nullable'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $field = "id";
        $type = "Asc";

        if ($request->order_by != null and $request->order_by != null) {
            if ($request->order_by == "rate") {
                $field = "rate";
                $type = "DESC";
            }
            if ($request->order_by == "desc") {
                $field = "id";
                $type = "DESC";
            }
        }
        $branches = Branch::with('vendor', 'area')->where('status', '!=', 'disable')->
        where(function ($query) use ($request) {
            if ($request->title != null and $request->title != " ") {
                $query->where('title', 'like', '%' . $request->title . '%');
            }
        })->where(function ($query) use ($request) {
            if ($request->area_id != null) {
                $query->whereHas('area', function ($query) use ($request) {
                    $query->where('id', $request->area_id);
                });
            }
        })->where(function ($query) use ($request) {
            if ($request->category_id != null) {
                $categoryChild = app(CategoryRepository::class)->getAllChildArray($request->category_id);
                $query->whereHas('category', function ($query) use ($categoryChild) {
                    $query->whereIn('id', $categoryChild);
                });
            }
        })->where(function ($query) use ($request) {
            if ($request->activity_id != null) {
                $query->whereIn("activities->*", $request->activity_id);
            }
        })->where(function ($query) use ($request) {
            if ($request->meals != null) {
                $query->whereJsonContains("meals", $request->meals);
            }
        })->where(function ($query) use ($request) {
            if ($request->lat != null and $request->lng != null) {
                $lat = $request->lat;
                $lng = $request->lng;
                $radius = 10;
                $haversine = "(6371 * acos(cos(radians($lat)) * cos(radians(lat)) * cos(radians(lng) - radians($lng)) + sin(radians($lat)) * sin(radians(lat))))";
                $query->selectRaw("{$haversine} AS distance")
                    ->whereRaw("{$haversine} < ?", [$radius]);
            }
        })->whereHas('products')->withCount('products')
            ->whereHas('productsWithVariety')->withAvg("productsWithVariety", 'price')
            ->withMin('productsWithVariety', 'price')->withMax('productsWithVariety', 'price')
            ->orderBy($field, $type);


        $countForPage = 10;
        if ($request->per_page and $request->per_page != null)
            $countForPage = $request->per_page;


        $branches=$branches->paginate((int)$countForPage);
        return response()->json(['data' => $branches], 200);
    }


    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'family' => 'required',
            'tel' => 'nullable',
            'mobile' => 'required',
            'national_code' => 'required',
            'user_name' => 'nullable',
            'birthday' => 'required',
            'referral_code' => 'nullable',
            'address' => 'required',
            'socials' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $data = $request->all();
        if (isset($data['socials'])) {
            $data['socials'] = json_encode($data['socials']);
        }
        $res = $this->vendorRep->register($data);
        if ($res == null)
            return response()->json(['data' => null], 500);
        return response()->json(['data' => $res], 200);
    }


}
