<?php

namespace App\Http\Controllers\V1\Admin;

use App\Enums\BranchStatus;
use App\Enums\ProductStatus;
use App\Http\Controllers\Controller;
use App\Models\Branch;
use App\Repositories\Area\AreaRepository;
use App\Repositories\Branch\BranchRepository;
use App\Repositories\User\UserRepository;
use App\Repositories\Vendor\VendorRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Enum;

class BranchController extends Controller
{
    protected $branchRepository;

    public function __construct()
    {
        $this->branchRepository = app(BranchRepository::class);
    }

    public function index(Request $request)
    {
        $data = $request->all();
        $data['is_admin'] = true;
        $data['sort_column'] = 'id';
        $data['sort_type'] = 'desc';
        Log::info("ds");
        $branches = $this->branchRepository->getWithFilter($data, true);
        return response()->json(['data' => $branches]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'address' => 'required',
            'lat' => 'required|numeric|between:-90,90',
            'lng' => 'required|numeric|between:-180,180',
            'category_id' => 'required|numeric|exists:categories,id',
            'area_id' => 'required|numeric|exists:areas,id',
            'agent_id' => 'nullable|numeric|exists:agents,id',
            'logo' => 'required|url',
            'cover' => 'required|url',
            'name' => 'required',
            'family' => 'required',
            'mobile' => ['required', Rule::unique('users')->where(function ($query) use ($request) {
                return $query->where('role_id', '!=', 4);
            })],

        ]);
        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->first(), 'status' => 400, 'data' => null], 400);
        }

        DB::beginTransaction();
        try {

            $user = app(UserRepository::class)->getUserWithMobile($request->mobile);
            if ($user and $user->role_id != 4) {
                return response()->json(['message' => 'شماره موبایل وارد شده دارای نقشی بجز کار میباشد', 'status' => 400, 'data' => null], 400);
            } elseif (!$user) {
                $user = app(UserRepository::class)->store(['name' => $request->name, 'family' => $request->family, 'mobile' => $request->mobile]);
            }

            $vendor = app(VendorRepository::class)->storeP(['title' => $request->title, 'user_id' => $user->id]);
            if ($vendor == null) {
                DB::rollback();
                return response()->json(['message' => 'در ایجاد مرکز با خطا مواجه شدیم', 'status' => 400, 'data' => null], 400);
            }

            $branch = $this->branchRepository->store([
                "title" => $request->title,
                "address" => $request->address,
                "sm_description" => $request->sm_description,
                "description" => $request->description,
                "lat" => $request->lat,
                "lng" => $request->lng,
                "category_id" => $request->category_id,
                "area_id" => $request->area_id,
                "logo" => $request->logo,
                "meals" => $request->meals,
                "activities" => $request->activities,
                "cover" => $request->cover,
                "vendor_id" => $vendor->id,
                "agent_id" => $request->agent_id ?? null,
            ]);
            if ($branch == null) {
                DB::rollback();
                return response()->json(['message' => 'در ایجاد مرکز با خطا مواجه شدیم', 'status' => 400, 'data' => null], 400);
            }

            $user = app(UserRepository::class)->update($user->id, ['role_id' => 3, 'target_role_id' => $branch->id]);
            if ($user == null) {
                DB::rollback();
                return response()->json(['message' => 'در ایجاد مرکز با خطا مواجه شدیم', 'status' => 400, 'data' => null], 400);
            }
            DB::commit();
            return response()->json(['data' => $branch, 'message' => null, 'status' => 201], 201);

        } catch (\Exception $e) {
            DB::rollback();
            handelError('error', $e->getMessage());
            return response()->json(['error' => 'عملیات با خطا مواجه شد', 'status' => 500, 'data' => null], 500);
        }

    }

    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $branch = $this->branchRepository->show($request->branch_id);

        if (!$branch) {
            return response()->json(['message' => 'برنچی یافت نشد'], 404);
        }
        return response()->json(['data' => $branch]);
    }

    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required',
            'address' => 'required',
            'branch_id' => 'required|numeric|exists:branches,id',
            'vendor_id' => 'nullable|numeric|exists:vendors,id',
            'lat' => 'required|numeric|between:-90,90',
            'lng' => 'required|numeric|between:-180,180',
            'category_id' => 'required|numeric|exists:categories,id',
            'area_id' => 'required|numeric|exists:areas,id',
            'agent_id' => 'nullable|numeric|exists:agents,id',
            'logo' => 'nullable|url',
            'cover' => 'required|url',
            'status' => [new Enum(BranchStatus::class)],

        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $data = $request->all();
        $branchId = $data['branch_id'];
        unset($data['branch_id']);
        $branch = $this->branchRepository->update($branchId, $data);
        if ($branch) {
            return response()->json(['message' => 'عملیات با موفقیت انجام شد'], 200);
        }
        return response()->json(['message' => 'عملیات با شکست مواجه شد'], 500);
    }

    public function destroy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $branch = $this->branchRepository->destroy($request->branch_id);
        if (!$branch) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 404);
        }
        return response()->json(['message' => 'شعبه با موفقیت حذف شد '], 200);
    }


    public function getBranchPhones(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $phones = $this->branchRepository->getBranchPhones($request->branch_id);
        if (!$phones) {
            return response()->json(['message' => 'تلفنی یافت نشد'], 404);
        }
        return response()->json(['data' => $phones], 200);
    }


    public function getGallery(Request $request)
    {
        $galleries = $this->branchRepository->getGalleryFirst($request->branch_id);
        return response()->json(['data' => $galleries], 200);

    }

    public function setGallery(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
            'images' => 'required|array',
            'images.*.path' => 'required|url',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $gallery = $this->branchRepository->getGallery($request->branch_id);
        if ($gallery == null || count($gallery) == 0) {
            $this->branchRepository->createGallery($request->branch_id, $request->images);
        } else {
            $this->branchRepository->updateGallery($request->branch_id, $request->images);
        }
        return response()->json(['message' => 'گالری با موفقیت ثبت شد'], 200);
    }


    public function getActivity(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $activities = $this->branchRepository->getActivityTime($request->branch_id);
        return response()->json(['data' => $activities], 200);
    }

    public function setActivity(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
            'to_time' => 'required',
            'from_time' => 'required',
            'week_day' => 'required',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $activities = $this->branchRepository->setActivityTime($request->all());
        if ($activities)
            return response()->json(['data' => $activities], 200);
        return response()->json(['message' => 'مشکلی پیش آمده است لطفا بعدا تلاش فرمایید'], 500);
    }

    public function destroyActivity(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
            'activity_id' => 'required|exists:active_times,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $activities = $this->branchRepository->destroyActivityTime($request->branch_id, $request->activity_id);
        if ($activities)
            return response()->json(['message' => 'زمان فعالیت با موفقیت حذف شد'], 200);
        return response()->json(['message' => 'مشکلی پیش آمده است لطفا بعدا تلاش فرمایید'], 500);
    }

    public function getBranchDeliveryAreas(Request $request)
    {
        $areas = app(BranchRepository::class)->getBranchDeliveryAreas(['branch_id' => $request->branch_id]);
        return response()->json(['data' => $areas['data']], $areas['status']);
    }


    public function setOrUpdateDeliveryArea(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
            'areas' => 'required|array',
            'areas.*.id' => 'required|exists:areas,id',
            'areas.*.shipment_price' => 'required',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $areas = app(BranchRepository::class)->setBranchArea($request->all());
        return response()->json(['data' => $areas], $areas['status']);
    }

    public function setOrUpdateDeliveryType($branchId, Request $request)
    {
        $validator = Validator::make($request->all(), [
            'deliveryTypes' => 'nullable|array',
            'deliveryTypes.*' => 'exists:delivery_types,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->first(), 'status' => 400, 'data' => []], 400);
        }
        $data = app(BranchRepository::class)->setBranchDeliveryTypes($branchId, $request->all()['deliveryTypes'] ?? []);
        return response()->json(["status" => 200, "message" => null, "data" => $data['data']], $data['status']);
    }

    public function getDeliveryType($branchId)
    {
        $data = app(BranchRepository::class)->getBranchDeliveryTypes($branchId);

        return response()->json(["status" => 200, "message" => null, "data" => $data['data']], $data['status']);
    }

    public function detachDeliveryArea(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
            'area_id' => 'required|exists:areas,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        app(BranchRepository::class)->detachBranchArea($request->all());
        return response()->json(['message' => 'عملیات با موفقیت انجام شد'], 200);
    }

}
