<?php

namespace App\Http\Controllers\V1\Admin;

use App\Enums\OrderStatus;
use App\Enums\TransactionType;
use App\Http\Controllers\Controller;
use App\Repositories\Branch\BranchRepository;
use App\Repositories\Order\OrderRepository;
use App\Repositories\Transaction\TransactionRepository;
use App\Repositories\User\UserRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Enum;


class OrderController extends Controller
{
    protected $orderRepository;

    public function __construct()
    {
        $this->orderRepository = app(OrderRepository::class);
    }

    public function index(Request $request)
    {

        Log:info("cdscsd");
        $categories = $this->orderRepository->getWithFilter($request->all());
        return response()->json(['data' => $categories], 200);
    }

    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric|exists:orders,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $order = $this->orderRepository->showA($request->order_id);
        if (!$order) {
            return response()->json(['message' => 'دسته بندی یافت نشد'], 404);
        }

        return response()->json(['data' => $order]);
    }

    public function changeStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric|exists:orders,id',
            'status' => [new Enum(OrderStatus::class)]
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $order = $this->orderRepository->changeStatus($request->order_id, $request->status);

        return response()->json($order,$order['status']);
    }

    public function acceptOrder(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric|exists:orders,id',
            'status' => [new Enum(OrderStatus::class)],
            'preparation_time' => 'required|numeric'
        ]);
        if ($validator->fails()) {
            return response()->json(['status' => '400', 'data' => [], 'message' => $validator->errors()->first()], 400);
        }
        return $this->orderRepository->acceptOrder($request->order_id, (int)$request->preparation_time);
    }

    public function getOrderForSend(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric|exists:orders,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $order = $this->orderRepository->getOrderForSend($request->order_id);
        if (!$order) {
            return response()->json(['message' => 'دسته بندی یافت نشد'], 404);
        }
        return response()->json(['message' => 'با موفقیت انجام شد']);
    }


    public function setDeliveryOrder(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric|exists:orders,id'
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $order = $this->orderRepository->setDeliveryOrder($request->order_id);
        return response()->json($order, $order['status'] ?? 500);
    }


    public function orderReturned(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required|numeric|exists:orders,id',
            'amount' => 'required',
            'description' => 'required',
            'whoseAccount' => 'required',
        ]);
        if ($validator->fails())
            return response()->json(['error' => $validator->errors()->first()], 400);

        $order = $this->orderRepository->show($request->order_id);
        if ($order->order_status != "delivered")
            return response()->json(['message' => 'عملیات مرجوعی فقط در صورت تکمیل شدن سفارش امکان پذیر است'], 400);

        $userBranch = app(UserRepository::class)->getUserWithBranchId($order->branch_id);
        $orderPrice = $order->order_price;
        $deliveryPrice = $order->delivery_price;
        $transactionRep = app(TransactionRepository::class);
        if ($request->whoseAccount == "branch") {
            $transactionRep->storeP(['amount' => $orderPrice * -1, 'user_id' => $userBranch->id, 'transaction_type' => TransactionType::RETURNED, 'description' => $request->description, 'target_id' => $order->id]);
            app(OrderRepository::class)->update($order->id, ['order_status' => OrderStatus::RETURNED]);
            return response()->json(['message' => 'با موفقیت انجام شد']);
        } elseif ($request->whoseAccount == "driver") {
            $transactionRep->storeP(['amount' => $orderPrice * -1, 'user_id' => $order->driver_id, 'transaction_type' => TransactionType::RETURNED, 'description' => $request->description, 'target_id' => $order->id]);
            app(OrderRepository::class)->update($order->id, ['order_status' => OrderStatus::RETURNED]);
            return response()->json(['message' => 'با موفقیت انجام شد']);
        } elseif ($request->whoseAccount == "branchAndDriver") {
            $transactionRep->storeP(['amount' => $orderPrice - $deliveryPrice * -1, 'user_id' => $userBranch->id, 'transaction_type' => TransactionType::RETURNED, 'description' => $request->description, 'target_id' => $order->id]);
            $transactionRep->storeP(['amount' => $deliveryPrice * -1, 'user_id' => $order->driver_id, 'transaction_type' => TransactionType::RETURNED, 'description' => $request->description, 'target_id' => $order->id]);
            app(OrderRepository::class)->update($order->id, ['order_status' => OrderStatus::RETURNED]);
            return response()->json(['message' => 'با موفقیت انجام شد']);
        } elseif ($request->whoseAccount == "system") {
            $transactionRep->storeP(['amount' => $orderPrice * -1, 'user_id' => 0, 'transaction_type' => TransactionType::RETURNEDSYSTEM, 'target_id' => $order->id, 'description' => $request->description]);
            return response()->json(['message' => 'با موفقیت انجام شد']);
        }
        return response()->json(['message' => 'عملیات با شکست مواجه شد'], 500);
    }
}
