<?php

namespace App\Http\Controllers\V1;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Repositories\Address\AddressRepository;
use App\Repositories\DeliveryType\DeliveryTypeRepository;
use App\Repositories\Order\OrderRepository;
use App\Repositories\User\UserRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class OrderController extends Controller
{
    protected $orderRep;
    protected $addressRep;

    public function __construct()
    {
        $this->orderRep = app(OrderRepository::class);
        $this->addressRep = app(AddressRepository::class);
    }

    public function getAllWithItems()
    {
        $order = $this->orderRep->getLatestOrderWithItems();
        if ($order['order'] instanceof Order)
            return response()->json(['data' => $order]);
        return response()->json(['data' => null], 404);
    }

    public function addProductCart(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'variety_id' => 'required|numeric|exists:varieties,id',
            'count' => 'required|numeric',
            'extraIds' => 'nullable|string',
            'time' => 'nullable|date_format:H:i:s',
            'week_day' => 'required|string|in:Sunday,Monday,Tuesday,Wednesday,Thursday,Friday,Saturday',
        ]);
        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->first(), 'status' => 400, 'data' => []], 400);
        }
        $order = $this->orderRep->addCartOrder($request->all());
        return response()->json(['data' => $order['data'], 'message' => $order['message'], 'status' => $order['status']], $order['status']);
    }

    public function storeOrUpdateOrderWithList(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'carts' => 'required|array',
            'carts.*.variety_id' => 'required|exists:varieties,id',
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }
        $this->orderRep->OrderListDeleteCart();
        foreach ($request->carts as $cart) {
            $order = $this->orderRep->addCartOrder($cart);
        }
        return response()->json(['data' => $order['data'], 'message' => $order['message'], 'status' => $order['status']], $order['status']);
    }

    public function minesProductCart(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'variety_id' => 'required|numeric|exists:varieties,id',
            'count' => 'required|numeric',
            'extraIds' => 'nullable|string',
        ]);
        if ($validator->fails()) {
            return response()->json(['message' => $validator->errors()->first(), 'status' => 400, 'data' => []], 400);
        }
        $order = $this->orderRep->minesCartOrder($request->all());
        return response()->json(['data' => $order['data'], 'message' => $order['message'], 'status' => $order['status']], $order['status']);
    }

    public function getCountItem()
    {
        $orderCount = $this->orderRep->getCountItem();
        return response()->json(['data' => $orderCount]);
    }

    public function deleteProductCart(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'variety_id' => 'required|numeric|exists:varieties,id',
            'count' => 'required|numeric',
            'extraIds' => 'nullable|string',
        ]);
        if ($validator->fails())
            return response()->json(['error' => $validator->errors()->first()], 400);
        $order = $this->orderRep->deleteCartOrder($request->all());
        if ($order['order'] instanceof Order)
            return response()->json(['data' => $order]);
        return response()->json(['data' => null], 404);
    }

    public function setDeliveryInfo(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'address_id' => 'nullable|numeric|exists:addresses,id',
        ]);
        if ($validator->fails())
            return response()->json(['error' => $validator->errors()->first()], 400);

        $order = $this->orderRep->getLatestOrderUser();

        if ($order->delivery_type_id == null)
            return response()->json(['status' => 400, 'message' => 'لطفا نحوه دریافت رو انتخاب کنید', 'data' => null], 400);

        if ($order->branch_id == null || $order->orderCart == null)
            return response()->json(['status' => 400, 'message' => 'اطلاعات نادرست است ', 'data' => null], 400);

        $check = app(DeliveryTypeRepository::class)->checkAccessAddress($order->id, $order->delivery_type_id, $request->address_id);

        if ($check['data']['check'] == false)
            return response()->json(['status' => 400, 'message' => $check['message'], 'data' => null], 400);

        $this->orderRep->update($order->id, ['address_id' => $request->address_id, 'delivery_type_id' => $order->delivery_type_id]);
        return response()->json(['status' => 200, 'message' => $check['message'] ?? 'آدرس با موفقیت ثبت شد', 'data' => $this->orderRep->getLatestOrderWithItems()], 200);
    }

    public function changeDeliveryType(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'delivery_type_id' => 'required|numeric|exists:delivery_types,id'
        ]);
        if ($validator->fails())
            return response()->json(['error' => $validator->errors()->first()], 400);

        $order = $this->orderRep->getLatestOrderUser();

        $deliveryType = app(DeliveryTypeRepository::class)->show($request->delivery_type_id);

        if ($order->branch_id == null || $order->orderCart == null || $deliveryType == null)
            return response()->json(['status' => 400, 'message' => 'اطلاعات نادرست است ', 'data' => null], 400);

        if (!$deliveryType->is_cost) {

            $price = $order->order_price - $order->delivery_price;
            $this->orderRep->update($order->id, ['delivery_price' => 0, 'order_price' => $price, 'delivery_type_id' => $request->delivery_type_id, 'delivery_type' => json_encode($deliveryType)]);

        } else {
            $this->orderRep->update($order->id, ['delivery_type_id' => $request->delivery_type_id, 'delivery_type' => json_encode($deliveryType)]);

        }


        return response()->json(['status' => 200, 'message' => $check['message'] ?? 'آدرس با موفقیت ثبت شد', 'data' => $this->orderRep->getLatestOrderWithItems()], 200);
    }

    public function checkTimeSet(Request $request)
    {
//        format date 1400/05/03 19:00:00
        $check = $this->addressRep->checkAddress($request->date, $request->branchId);
        return response()->json(['data' => $check['data'] ?? null, 'message' => $check['message']], $check['status']);
    }

    public function getListAddressAccessCart()
    {
        $address = $this->orderRep->getListAddressAccessCart();
        return response()->json(['data' => $address]);
    }

    public function orderListBefore()
    {
        $orders = $this->orderRep->orderListBefore();
        return response()->json(['data' => $orders]);
    }

    public function getListGateways(Request $request)
    {
        $data = $this->orderRep->getListGateways();
        return response()->json(['data' => $data]);
    }

    public function checkCoupon(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'coupon' => 'required'
        ]);
        if ($validator->fails())
            return response()->json(['error' => $validator->errors()->first()], 400);
        $order = $this->orderRep->checkCoupon($request->all());
        return response()->json($order, $order['status']);
    }

    public function getOrderUser(Request $request)
    {
        $orders = app(UserRepository::class)->getOrders($request->all());
        if (!$orders) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 500);
        }
        return response()->json(['data' => $orders]);
    }

    public function getOrder(Request $request, $id)
    {
        $orders = $this->orderRep->showWithUser($id);
        if (!$orders) {
            return response()->json(['message' => 'عملیات با شکست مواجه شد'], 500);
        }
        return response()->json(['data' => $orders]);
    }

    public function createComment(Request $request, $id)
    {
        $orders = $this->orderRep->createComment($request->all(), $id);
        return response()->json($orders, $orders['status']);
    }


}
