<?php

namespace App\Listeners;

use App\Enums\NotificationType;
use App\Enums\OrderStatus;
use App\Events\OrderChangeStatus;
use App\Models\NotificationMobile;
use App\Repositories\Branch\BranchRepository;
use App\Repositories\SMS\SMSRepositoryInterface;
use App\Repositories\User\UserRepository;
use Defuse\Crypto\RuntimeTests;
use Illuminate\Support\Facades\Log;

class OrderChangeStatusSendSms
{
    protected $SMSRepository;
    protected $userBranch;
    protected $branch;

    public function __construct(SMSRepositoryInterface $SMSRepository)
    {
        $this->SMSRepository = $SMSRepository;
    }

    public function handle(OrderChangeStatus $event)
    {
        $order = $event->order;
        $user = app(UserRepository::class)->show($order->user_id);
        $userBranch = $this->userBranch = app(UserRepository::class)->getUserWithBranchId($order->branch_id);
        $branch = $this->branch = app(BranchRepository::class)->show($order->branch_id);
        // Add logic to send SMS to the driver
        $this->sendSMSToDriver($order);

        // Add logic to send SMS to the user
        $this->sendSMSToUser($order, $user, $branch);

        // Add logic to send SMS to the restaurant
        $this->sendSMSToRestaurant($order, $user, $userBranch);
    }

    private function sendSMSToDriver($order)
    {
        if ($order->order_status == OrderStatus::READYTOSEND->value) {
            $drivers = app(UserRepository::class)->getDriversMobileNumber();
            foreach ($drivers as $driver) {
                $result = $this->SMSRepository->sendSMSToDriver('#' . $order->id, $driver);
            }
        }
    }

    private function sendSMSToRestaurant($order, $user, $userBranch)
    {
        if ($order->order_status == OrderStatus::AWAITINGRESTAURANTAPPRAVAL) {
            $notificationMobiles = NotificationMobile::where('branch_id', $order->branch_id)
                ->where('type', NotificationType::ORDER_NEW)
                ->where('is_active', true)
                ->pluck('mobile')
                ->toArray();

            if (!empty($notificationMobiles)) {
                foreach ($notificationMobiles as $mobile) {
                    $result = $this->SMSRepository->sendSMSNewOrder('#' . $order->id, $mobile);
                }
            } else {
                if ($userBranch && $userBranch->mobile) {
                    $result = $this->SMSRepository->sendSMSNewOrder('#' . $order->id, $userBranch->mobile);
                }
            }

            event(new \App\Events\NotificationOrder($order));
        }
    }

    private function sendSMSToUser($order, $user, $branch)
    {
        if ($order->order_status == OrderStatus::AWAITINGRESTAURANTAPPRAVAL->value) {
            $result = $this->SMSRepository->sendSMSToUserAwaitingRestaurantApproval('#' . $order->id, $user->mobile);
        }
        if ($order->order_status == OrderStatus::PREPARING->value) {
            $result = $this->SMSRepository->sendSMSOrderStatus('#' . $order->id, 'در حال آماده سازی', $user->mobile);
        }
        Log::info($order);

        if ($order->order_status == OrderStatus::SENDING->value) {
            $result = $this->SMSRepository->sendSMSOrderStatus('#' . $order->id, 'در حال ارسال', $user->mobile);
        }
        if ($order->order_status == OrderStatus::DELIVERED->value) {
            $amount = app(UserRepository::class)->getWalletUserInBranch($branch->id, $order->user_id);
            if ($amount > 0)
                $result = $this->SMSRepository->sendSMSOrderDeliveredWithCashBack($user->name ?? ' عزیز ', $amount, $user->mobile);
            else
                $result = $this->SMSRepository->sendSMSOrderStatus('#' . $order->id, 'تحویل داده شد', $user->mobile);
        }
    }
}
