<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class AreaSpecialCondition extends Model
{
    use HasFactory;

    protected $guarded = ['id'];

    protected $casts = [
        'percent' => 'integer',
        'branch_id' => 'array',
        'area_id' => 'array',
    ];

    /**
     * Get the start time in H:i format
     */
    public function getStartTimeAttribute($value)
    {
        return $value ? date('H:i', strtotime($value)) : null;
    }

    /**
     * Get the end time in H:i format
     */
    public function getEndTimeAttribute($value)
    {
        return $value ? date('H:i', strtotime($value)) : null;
    }

    /**
     * Set the start time from H:i format
     */
    public function setStartTimeAttribute($value)
    {
        $this->attributes['start_time'] = $value;
    }

    /**
     * Set the end time from H:i format
     */
    public function setEndTimeAttribute($value)
    {
        $this->attributes['end_time'] = $value;
    }

    /**
     * Get the branches that are associated with this special condition.
     */
    public function branches()
    {
        if (empty($this->branch_id) || !is_array($this->branch_id)) {
            return collect([]);
        }
        return Branch::whereIn('id', $this->branch_id)
            ->select('id', 'title', 'status', 'address', 'min_order', 'logo')
            ->with('phones:id,branch_id,tel')
            ->get()
            ->map(function ($branch) {
                return [
                    'id' => $branch->id,
                    'title' => $branch->title,
                    'status' => $branch->status,
                    'address' => $branch->address,
                    'min_order' => $branch->min_order,
                    'logo' => $branch->logo,
                    'phones' => $branch->phones->pluck('tel')->toArray()
                ];
            });
    }

    /**
     * Get the areas that are associated with this special condition.
     */
    public function areas()
    {
        if (empty($this->area_id) || !is_array($this->area_id)) {
            return collect([]);
        }
        return Area::whereIn('id', $this->area_id)
            ->select('id', 'title', 'parent_id')
            ->with('parent:id,title')
            ->get()
            ->map(function ($area) {
                return [
                    'id' => $area->id,
                    'title' => $area->title,
                    'parent_id' => $area->parent_id,
                    'parent_title' => $area->parent ? $area->parent->title : null
                ];
            });
    }

    /**
     * Scope to get active conditions
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope to get conditions for a specific branch
     */
    public function scopeForBranch($query, $branchId)
    {
        return $query->whereJsonContains('branch_id', $branchId);
    }

    /**
     * Scope to get conditions for a specific area
     */
    public function scopeForArea($query, $areaId)
    {
        return $query->whereJsonContains('area_id', $areaId);
    }

    /**
     * Scope to get conditions for a specific date
     */
    public function scopeForDate($query, $date)
    {
        return $query->where('date', $date);
    }

    /**
     * Scope to get conditions within time range
     */
    public function scopeWithinTimeRange($query, $startTime, $endTime)
    {
        return $query->where(function ($q) use ($startTime, $endTime) {
            $q->whereBetween('start_time', [$startTime, $endTime])
              ->orWhereBetween('end_time', [$startTime, $endTime])
              ->orWhere(function ($subQ) use ($startTime, $endTime) {
                  $subQ->where('start_time', '<=', $startTime)
                       ->where('end_time', '>=', $endTime);
              });
        });
    }
}
