<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Branch extends Model
{
    use HasFactory;
    use SoftDeletes;

    protected $guarded = ['id'];
    protected $casts = [
        'per_order_status' => 'boolean',
    ];

    public function area()
    {
        return $this->belongsTo(Area::class, 'area_id', 'id');
    }

    public function category()
    {
        return $this->belongsTo(Category::class, 'category_id', 'id');
    }

    public function vendor()
    {
        return $this->belongsTo(Vendor::class, 'vendor_id', 'id');
    }

    /**
     * Get the user (owner) of this branch.
     * User with role_id=2 (vendor) and target_role_id=branch_id
     */
    public function user()
    {
        return $this->hasOne(User::class, 'target_role_id', 'id')->where('role_id', 2);
    }

    /**
     * Get the bank accounts for this branch through user relationship.
     */
    public function bankAccounts()
    {
        return $this->hasManyThrough(
            BankAccount::class,
            User::class,
            'target_role_id', // Foreign key on users table
            'user_id',        // Foreign key on bank_accounts table
            'id',             // Local key on branches table
            'id'              // Local key on users table
        )->where('users.role_id', 2);
    }

    public function gallery()
    {
        return $this->hasMany(Gallery::class, 'branch_id', 'id')->with('files');
    }

    public function products()
    {
        return $this->hasMany(Product::class, 'branch_id', 'id');
    }

    public function productsWithVariety()
    {
        return $this->belongsToMany(Variety::class, 'products', 'branch_id', 'id', 'id', 'product_id');
    }

    public function deliveryAreas()
    {
        return $this->belongsToMany(Area::class, 'branch_delivery_areas', 'branch_id', 'area_id')
            ->withPivot('shipment_price');
    }

    public function deliveryTypes()
    {
        return $this->belongsToMany(DeliveryType::class, 'branch_delivery_types', 'branch_id', 'delivery_type_id');
    }

    public function getProductSortMinimumPrice()
    {
        return $this->belongsToMany(Variety::class, 'products', 'branch_id', 'id', 'id', 'product_id')
            ->orderBy('price');
    }

    public function phones()
    {
        return $this->hasMany(Phone::class, 'branch_id', 'id');
    }

    /**
     * Get the customer branches for this branch.
     */
    public function customerBranches()
    {
        return $this->hasMany(CustomerBranch::class);
    }

    /**
     * Get the customers for this branch.
     */
    public function customers()
    {
        return $this->belongsToMany(User::class, 'customer_branches', 'branch_id', 'user_id')
                    ->withPivot(['name', 'family', 'mobile', 'email', 'address', 'status']);
    }


    public function otherBranch()
    {
        return $this->hasMany(Branch::class, 'vendor_id', 'vendor_id')->where('id', '!=', $this->id);
    }

    public function openingHours()
    {
        return $this->hasMany(ActiveTime::class, 'branch_id', 'id');
    }

    /**
     * Get the holidays for this branch.
     */
    public function holidays()
    {
        return $this->hasMany(BranchHoliday::class, 'branch_id', 'id');
    }

    /**
     * Get active holidays for this branch.
     */
    public function activeHolidays()
    {
        return $this->hasMany(BranchHoliday::class, 'branch_id', 'id')->active();
    }

    public function nearestOpeningHour()
    {
        $current_time = Carbon::now()->format('H:i:s');
        $current_day = Carbon::now()->dayOfWeek;

        $query = $this->hasOne(ActiveTime::class, 'branch_id', 'id')
            ->where(function ($q) use ($current_day, $current_time) {
                $q->where('week_day', $current_day)
                    ->where('from_time', '>', $current_time);
            })
            ->orWhere(function ($q) use ($current_day) {
                $q->where('week_day', '>', $current_day);
            })
            ->orderByRaw("FIELD(week_day, $current_day, $current_day + 1, $current_day + 2, $current_day + 3, $current_day + 4, $current_day + 5, $current_day + 6)")
            ->orderBy('from_time', 'asc');

        return $query;
    }

    public function areaManagements()
    {
        return $this->hasMany(AreaManagement::class, 'branch_id');
    }

    /**
     * Get the popup alarms for this branch.
     */
    public function popupAlarms()
    {
        return $this->hasMany(PopupAlarm::class, 'branch_id');
    }

    /**
     * Get active popup alarms for this branch.
     */
    public function activePopupAlarms()
    {
        return $this->hasMany(PopupAlarm::class, 'branch_id')
                    ->where('status', 'active')
                    ->where('start_time', '<=', now())
                    ->where('end_time', '>=', now());
    }

}
