<?php

namespace App\Models;

use App\Enums\HolidayReasonType;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class BranchHoliday extends Model
{
    use HasFactory;

    protected $fillable = [
        'branch_id',
        'holiday_type',
        'start_date',
        'end_date',
        'start_time',
        'end_time',
        'is_full_day',
        'week_days',
        'repeat_yearly',
        'reason',
        'holiday_reason_type',
        'status'
    ];

    protected $casts = [
        'start_time' => 'datetime:H:i',
        'end_time' => 'datetime:H:i',
        'is_full_day' => 'boolean',
        'week_days' => 'array',
        'repeat_yearly' => 'boolean',
        'holiday_reason_type' => HolidayReasonType::class,
        'status' => 'string'
    ];

    /**
     * رابطه با شعبه
     */
    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }

    /**
     * بررسی اینکه آیا تاریخ مشخص در تعطیلی قرار دارد یا نه
     */
    public function isDateInHoliday($date)
    {
        $checkDate = $date instanceof Carbon ? $date : Carbon::parse($date);

        switch ($this->holiday_type) {
            case 'single_date':
                return $checkDate->isSameDay($this->start_date);

            case 'date_range':
                if ($this->end_date) {
                    return $checkDate->between($this->start_date, $this->end_date);
                }
                return $checkDate->isSameDay($this->start_date);

            case 'weekly_pattern':
                if ($this->week_days && is_array($this->week_days)) {
                    $dayName = strtolower($checkDate->format('l'));
                    return in_array($dayName, $this->week_days);
                }
                return false;

            default:
                return false;
        }
    }

    /**
     * بررسی اینکه آیا زمان مشخص در تعطیلی قرار دارد یا نه
     */
    public function isTimeInHoliday($date, $time = null)
    {
        if (!$this->isDateInHoliday($date)) {
            return false;
        }

        if ($this->is_full_day) {
            return true;
        }

        if (!$this->start_time || !$this->end_time) {
            return false;
        }

        $checkTime = $time ? Carbon::parse($time)->format('H:i:s') : Carbon::now()->format('H:i:s');

        return $checkTime >= $this->start_time->format('H:i:s') &&
               $checkTime <= $this->end_time->format('H:i:s');
    }

    /**
     * بررسی تعطیلی‌های فعال
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * فیلتر بر اساس نوع تعطیلی
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('holiday_type', $type);
    }

    /**
     * فیلتر بر اساس شعبه
     */
    public function scopeForBranch($query, $branchId)
    {
        return $query->where('branch_id', $branchId);
    }

    /**
     * فیلتر بر اساس تاریخ
     */
    public function scopeForDate($query, $date)
    {
        $checkDate = $date instanceof Carbon ? $date : Carbon::parse($date);

        return $query->where(function ($q) use ($checkDate) {
            $q->where('start_date', '<=', $checkDate)
              ->where(function ($subQ) use ($checkDate) {
                  $subQ->whereNull('end_date')
                        ->orWhere('end_date', '>=', $checkDate);
              });
        });
    }

    /**
     * فیلتر بر اساس نوع علت تعطیلی
     */
    public function scopeByReasonType($query, HolidayReasonType $reasonType)
    {
        return $query->where('holiday_reason_type', $reasonType);
    }

    /**
     * دریافت لیبل فارسی نوع علت تعطیلی
     */
    public function getHolidayReasonTypeLabelAttribute(): ?string
    {
        return $this->holiday_reason_type?->getLabel();
    }

    /**
     * بررسی اینکه آیا تعطیلی دارای نوع علت مشخص است یا نه
     */
    public function hasReasonType(): bool
    {
        return !is_null($this->holiday_reason_type);
    }
}
