<?php

namespace App\Models;

use App\Enums\NotificationType;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class NotificationMobile extends Model
{
    use HasFactory;

    protected $fillable = [
        'branch_id',
        'type',
        'mobile',
        'is_active'
    ];

    protected $casts = [
        'type' => NotificationType::class,
        'is_active' => 'boolean',
    ];

    /**
     * Get the branch that owns the notification mobile.
     */
    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    /**
     * Scope to get active notification mobiles
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to filter by branch
     */
    public function scopeForBranch($query, $branchId)
    {
        return $query->where('branch_id', $branchId);
    }

    /**
     * Scope to filter by type
     */
    public function scopeOfType($query, NotificationType $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Get mobile numbers for a specific branch and type
     */
    public static function getMobilesForBranchAndType($branchId, NotificationType $type): array
    {
        return self::active()
            ->forBranch($branchId)
            ->ofType($type)
            ->pluck('mobile')
            ->toArray();
    }

    /**
     * Get all mobile numbers for a specific type (across all branches)
     */
    public static function getMobilesForType(NotificationType $type): array
    {
        return self::active()
            ->ofType($type)
            ->pluck('mobile')
            ->toArray();
    }
}
