<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Morilog\Jalali\Jalalian;
use App\Enums\PopupAlarmPageType;
use Illuminate\Database\Eloquent\SoftDeletes;

class PopupAlarm extends Model
{
    use HasFactory;
    use SoftDeletes;

    protected $fillable = [
        'branch_id',
        'title',
        'message',
        'image',
        'start_time',
        'end_time',
        'status',
        'page_type'
    ];

    protected $casts = [
        'start_time' => 'datetime',
        'end_time' => 'datetime',
        'page_type' => PopupAlarmPageType::class,
    ];

    /**
     * رابطه با رستوران
     */
    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    /**
     * رابطه با نمایش‌های آلارم
     */
    public function views(): HasMany
    {
        return $this->hasMany(PopupAlarmView::class);
    }

    /**
     * بررسی اینکه آیا آلارم فعال است یا نه
     */
    public function isActive(): bool
    {
        $now = now();
        return $this->status === 'active' &&
               $this->start_time <= $now &&
               $this->end_time >= $now;
    }

    /**
     * دریافت تاریخ شمسی شروع
     */
    public function getStartTimeJalaliAttribute(): string
    {
        return Jalalian::fromCarbon($this->start_time)->format('Y/m/d H:i');
    }

    /**
     * دریافت تاریخ شمسی پایان
     */
    public function getEndTimeJalaliAttribute(): string
    {
        return Jalalian::fromCarbon($this->end_time)->format('Y/m/d H:i');
    }

    /**
     * دریافت نام صفحه نمایش
     */
    public function getPageTypeLabelAttribute(): string
    {
        return $this->page_type->getLabel();
    }
}
