<?php

namespace App\Providers;

use App\Repositories\Print\PrintRepository;
use App\Repositories\SMS\SMSRepositoryInterface;
use App\Repositories\SMS\ChainSMSRepository;

use App\Repositories\SMS\FarazSmsRepository;
use App\Repositories\SMS\KavenegarRepository;
use App\Repositories\SMS\MelipayamakDedicatedNumberRepository;
use App\Repositories\SMS\MelipayamakRepository;

use App\Repositories\AreaSpecialCondition\RepositoryInterface as AreaSpecialConditionRepositoryInterface;
use App\Repositories\AreaSpecialCondition\EloquentRepository as AreaSpecialConditionEloquentRepository;

use Illuminate\Cache\RateLimiting\Limit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\ServiceProvider;
use Laravel\Passport\Passport;

class AppServiceProvider extends ServiceProvider
{
    public function register(): void
    {
        // 1) از دیتابیس سرویس‌های فعال رو به ترتیب اولویت می‌خونیم
        $this->app->singleton(SMSRepositoryInterface::class, function ($app) {
            $map = [
                'FARAZSMS'          => FarazSmsRepository::class,
                'KAVENEGAR'         => KavenegarRepository::class,
                'MELIPAYAMAKNUMBER' => MelipayamakDedicatedNumberRepository::class,
                'MELIPAYAMAK'       => MelipayamakRepository::class,
            ];

            $rows = DB::table('sms_settings')
                ->where('status', 1)
                ->orderBy('order')
                ->get();

            $providers = [];
            foreach ($rows as $row) {
                $key = strtoupper((string)$row->repository);
                $class = $map[$key] ?? null;
                if ($class) {
                    $providers[] = $app->make($class); // تزریق وابستگی‌ها حفظ می‌شود
                } else {
                    Log::warning("Unknown SMS repository key in sms_settings: {$row->repository}");
                }
            }

            // اگر دیتابیس خالی بود، حداقل یک fallback داشته باش
            if (empty($providers)) {
                $providers[] = $app->make(MelipayamakRepository::class);
            }

            // برگردوندن چِین که خودش اتومات می‌افته سراغ سرویس بعدی
            return new ChainSMSRepository($providers);
        });

        $this->app->bind(PrintRepository::class, PrintRepository::class);
        $this->app->bind(
            \Modules\Courier\Repositories\BranchCourierRepositoryInterface::class,
            \Modules\Courier\Repositories\BranchCourierRepository::class
        );

        $this->app->bind(
            AreaSpecialConditionRepositoryInterface::class,
            AreaSpecialConditionEloquentRepository::class
        );

        // Bind user order repository interface to implementation
        $this->app->bind(
            \App\Repositories\UserOrder\UserOrderRepositoryInterface::class,
            \App\Repositories\UserOrder\UserOrderRepository::class
        );

        // Bind popup alarm repository interface to implementation
        $this->app->bind(
            \App\Repositories\PopupAlarm\RepositoryInterface::class,
            \App\Repositories\PopupAlarm\PopupAlarmRepository::class
        );

        // Bind notification mobile repository interface to implementation
        $this->app->bind(
            \App\Repositories\NotificationMobile\NotificationMobileRepositoryInterface::class,
            \App\Repositories\NotificationMobile\NotificationMobileRepository::class
        );
    }

    public function boot(): void
    {
        RateLimiter::for('api', function (Request $request) {
            return Limit::perMinute(100);
        });

        Schema::defaultStringLength(191);

        Passport::tokensExpireIn(now()->addDays(15));
        Passport::refreshTokensExpireIn(now()->addDays(30));
        Passport::personalAccessTokensExpireIn(now()->addMonths(6));
    }
}
