<?php

namespace App\Repositories\AreaManagment;

use App\Models\AreaManagement;
use App\Models\AreaManagementStatus;
use App\Models\Address;
use Illuminate\Support\Facades\Log;

class AreaManagmentRepository implements RepositoryInterface
{
    protected $model;
    protected $areaManagementStatus;
    public function __construct(AreaManagement $model, AreaManagementStatus $status)
    {
        $this->model = $model;
        $this->areaManagementStatus = $status;
    }

    /**
     * Check if area management status is active
     */
    private function isAreaManagementStatusActive($areaId, $branchId)
    {
        $areaStatus = $this->areaManagementStatus
            ->where('area_id', $areaId)
            ->where('branch_id', $branchId)
            ->first();

        return $areaStatus ? $areaStatus->status : true; // Default to true if no status record exists
    }

    /**
     * Get shipping method names by IDs
     */
    private function getShippingMethodNames($methodIds)
    {
        if (empty($methodIds)) {
            return [];
        }

        $deliveryTypes = \App\Models\DeliveryType::whereIn('id', $methodIds)->get();
        $methodNames = [];

        foreach ($methodIds as $methodId) {
            $deliveryType = $deliveryTypes->firstWhere('id', $methodId);
            $methodNames[] = $deliveryType ? $deliveryType->title : $methodId;
        }

        return $methodNames;
    }

    public function createManagement(array $data)
    {
        $schedules = $data['schedules'];
        $areaId = $data['area_id'];
        $branchId = $data['branch_id'];


        $createdSchedules = [];

        foreach ($schedules as $schedule) {
            foreach ($schedule['day'] as $day) {
            $scheduleData = [
                'area_id' => $areaId,
                'branch_id' => $branchId,
                'day' => $day,
                'service_time_from' => $schedule['service_time_from'],
                'service_time_to' => $schedule['service_time_to'],
                'min_order' => $schedule['min_order'],
                'delivery_time' => $schedule['delivery_time']??0,
                'order_limit_per_half_hour' => $schedule['order_limit_per_half_hour']??0,
                'free_delivery' => $schedule['free_delivery']??false,
                'free_delivery_min_amount' => $schedule['free_delivery_min_amount']??0,
                'status' => $schedule['status']??false,
                'out_of_range_price' => $schedule['out_of_range_price']??0,
                'order_amounts' => $schedule['order_amounts']??[],
                'shipping_methods' => $schedule['shipping_methods']??[],
                ];
                $createdSchedules[] = $this->model->create($scheduleData);
            }
        }

        return $createdSchedules;
    }

    public function updateManagement(array $data)
    {
        Log::info('Updating area management', ['area_id' => $data['area_id'], 'branch_id' => $data['branch_id']]);
        $schedules = $data['schedules'];
        $areaId = $data['area_id'];
        $branchId = $data['branch_id'];

        // Delete all existing schedules for this area and branch
        $this->model
            ->where('area_id', $areaId)
            ->where('branch_id', $branchId)
            ->delete();

        $updatedSchedules = [];

        foreach ($schedules as $schedule) {
            // Handle days as array (for checkbox selection)
            $days = is_array($schedule['day']) ? $schedule['day'] : [$schedule['day']];

            foreach ($days as $day) {
                // Create new schedule for each day
                $scheduleData = [
                    'area_id' => $areaId,
                    'branch_id' => $branchId,
                    'day' => $day,
                    'service_time_from' => $schedule['service_time_from'],
                    'service_time_to' => $schedule['service_time_to'],
                    'min_order' => $schedule['min_order'],
                    'delivery_time' => $schedule['delivery_time'] ?? 0,
                    'order_limit_per_half_hour' => $schedule['order_limit_per_half_hour'] ?? 0,
                    'free_delivery' => $schedule['free_delivery'] ?? false,
                    'free_delivery_min_amount' => $schedule['free_delivery_min_amount'] ?? null,
                    'status' => $schedule['status'] ?? false,
                    'out_of_range_price' => $schedule['out_of_range_price'] ?? 0,
                    'order_amounts' => $schedule['order_amounts'] ?? [],
                    'shipping_methods' => $schedule['shipping_methods'] ?? [],
                ];

                $updatedSchedules[] = $this->model->create($scheduleData);
            }
        }

        return $updatedSchedules;
    }

    public function getBranchAreas($branchId)
    {
        $areaManagements = $this->model
            ->with(['area'])
            ->where('branch_id', $branchId)
            ->get()
            ->groupBy('area_id');

        return $areaManagements->map(function ($schedules, $areaId) use ($branchId) {
            $firstSchedule = $schedules->first();
            $shipmentTypes = $firstSchedule->shipping_methods;
            $title = $firstSchedule->area->title??'-';
            $coordinates = $firstSchedule->area->coordinates ?? null;

            // Get area management status
            $areaManagementStatus = $this->isAreaManagementStatusActive($areaId, $branchId);

            return [
                'schedules' => $schedules->map(function ($schedule) {
                    return [
                        'id' => $schedule->id,
                        'day' => $schedule->day,
                        'service_time_from' => $schedule->service_time_from,
                        'service_time_to' => $schedule->service_time_to,
                        'min_order' => $schedule->min_order,
                        'delivery_time' => $schedule->delivery_time,
                        'order_limit_per_half_hour' => $schedule->order_limit_per_half_hour,
                        'free_delivery' => $schedule->free_delivery,
                        'free_delivery_min_amount' => $schedule->free_delivery_min_amount,
                        'status' => $schedule->status,
                        'out_of_range_price' => $schedule->out_of_range_price,
                        'order_amounts' => $schedule->order_amounts,
                        'shipping_methods' => $this->getShippingMethodNames($schedule->shipping_methods),
                    ];
                })->values()->toArray(),
                'area_id' => $areaId,
                'title' => $title,
                'branch_id' => $branchId,
                'shipping_methods' => $this->getShippingMethodNames($shipmentTypes),
                'coordinates' => $coordinates,
                'area_management_status' => $areaManagementStatus,
            ];
        })->values();
    }

    public function showManagement(array $data)
    {
        $schedules = $this->model
            ->with(['area'])
            ->where('branch_id', $data['branch_id'])
            ->where('area_id', $data['area_id'])
            ->get();

        if ($schedules->isEmpty()) {
            return null;
        }

        $firstSchedule = $schedules->first();
        $shipmentTypes = $firstSchedule->shipping_methods;
        $title = $firstSchedule->area->title??'-';

        // Group schedules by time range and combine days
        $groupedSchedules = [];
        foreach ($schedules as $schedule) {
            $key = $schedule->service_time_from . '_' . $schedule->service_time_to . '_' . $schedule->min_order;

                if (!isset($groupedSchedules[$key])) {
                $groupedSchedules[$key] = [
                    'id' => $schedule->id, // Keep the first ID for editing
                    'day' => [$schedule->day],
                    'service_time_from' => $schedule->service_time_from,
                    'service_time_to' => $schedule->service_time_to,
                    'min_order' => $schedule->min_order,
                    'delivery_time' => $schedule->delivery_time,
                    'order_limit_per_half_hour' => $schedule->order_limit_per_half_hour,
                    'free_delivery' => $schedule->free_delivery,
                    'free_delivery_min_amount' => $schedule->free_delivery_min_amount,
                    'status' => $schedule->status,
                    'out_of_range_price' => $schedule->out_of_range_price,
                    'order_amounts' => $schedule->order_amounts,
                        // For edit forms we must return IDs to correctly pre-check checkboxes
                        'shipping_methods' => $schedule->shipping_methods,
                        // Provide human-readable titles separately if needed by clients
                        'shipping_methods_titles' => $this->getShippingMethodNames($schedule->shipping_methods),
                ];
            } else {
                // Add the day to the existing group
                $groupedSchedules[$key]['day'][] = $schedule->day;
            }
        }

        return [
            'schedules' => array_values($groupedSchedules),
            'area_id' => $data['area_id'],
            'title' => $title,
            'branch_id' => $data['branch_id'],
            // Keep titles at top-level for display, IDs are inside each schedule
            'shipping_methods_titles' => $this->getShippingMethodNames($shipmentTypes),
            'area' => $firstSchedule->area
        ];
    }

    public function deleteManagement(array $data)
    {
        $deleted = $this->model
            ->where('branch_id', $data['branch_id'])
            ->where('area_id', $data['area_id'])
            ->delete();

        return $deleted > 0;
    }


    public function getActiveByBranchDayTime($branchId, $day, $currentTime)
    {
        $activeAreas = $this->model
            ->where('branch_id', $branchId)
            ->where('day', $day)
            ->where('status', true)
            ->where('service_time_from', '<=', $currentTime)
            ->where('service_time_to', '>=', $currentTime)
            ->get();

        // Filter out areas where area management status is disabled
        return $activeAreas->filter(function ($area) {
            return $this->isAreaManagementStatusActive($area->area_id, $area->branch_id);
        });
    }

    public function getActiveByBranch($branchId)
    {
        $activeAreas = $this->model
            ->where('branch_id', $branchId)
            ->where('status', true)
            ->get();

        // Filter out areas where area management status is disabled
        return $activeAreas->filter(function ($area) {
            return $this->isAreaManagementStatusActive($area->area_id, $area->branch_id);
        });
    }

    /**
     * find the next available time for a branch
     * @param int $branchId
     * @param string $currentDay
     * @param string $currentTime
     * @return AreaManagement|null
     */
    public function getNextAvailableByBranch($branchId, $currentDay, $currentTime)
    {

        $daysOfWeek = [
            'Saturday', 'Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday'
        ];
        $currentDayIndex = array_search($currentDay, $daysOfWeek);
        if ($currentDayIndex === false) {
            $currentDayIndex = 0;
        }


        $todaySchedules = $this->model
            ->where('branch_id', $branchId)
            ->where('day', $currentDay)
            ->where('status', true)
            ->where('service_time_from', '>', $currentTime)
            ->orderBy('service_time_from', 'asc')
            ->get();

        // Filter out areas where area management status is disabled
        $activeTodaySchedules = $todaySchedules->filter(function ($area) {
            return $this->isAreaManagementStatusActive($area->area_id, $area->branch_id);
        });

        if ($activeTodaySchedules->count() > 0) {
            // First try to find area with area_id != 0, if not found return area with area_id = 0
            $areaWithNonZeroId = $activeTodaySchedules->where('area_id', '!=', 0)->first();
            $areaWithZeroId = $activeTodaySchedules->where('area_id', 0)->first();

            return $areaWithNonZeroId ?: $areaWithZeroId;
        }


        for ($i = 1; $i < 7; $i++) {
            $nextDayIndex = ($currentDayIndex + $i) % 7;
            $nextDay = $daysOfWeek[$nextDayIndex];
            $nextDaySchedules = $this->model
                ->where('branch_id', $branchId)
                ->where('day', $nextDay)
                ->where('status', true)
                ->orderBy('service_time_from', 'asc')
                ->get();

            // Filter out areas where area management status is disabled
            $activeNextDaySchedules = $nextDaySchedules->filter(function ($area) {
                return $this->isAreaManagementStatusActive($area->area_id, $area->branch_id);
            });

            if ($activeNextDaySchedules->count() > 0) {
                // First try to find area with area_id != 0, if not found return area with area_id = 0
                $areaWithNonZeroId = $activeNextDaySchedules->where('area_id', '!=', 0)->first();
                $areaWithZeroId = $activeNextDaySchedules->where('area_id', 0)->first();

                return $areaWithNonZeroId ?: $areaWithZeroId;
            }
        }

        return null;
    }


    public function checkClockActivity($areaId, $branchId, $weekDay = null, $time = null)
    {
        if ($weekDay == null)
            $weekDay = now()->format('l');
        if ($time == null)
            $time = now()->format('H:i:s');

        $areaManagement = $this->model
            ->where('area_id', $areaId)
            ->where('branch_id', $branchId)
            ->where('day', $weekDay)
            ->where('status', true)
            ->where('service_time_from', '<=', $time)
            ->where('service_time_to', '>=', $time)
            ->first();

        // Check if area management status is active
        if ($areaManagement && !$this->isAreaManagementStatusActive($areaId, $branchId)) {
            return null;
        }

        return $areaManagement;
    }

    public function toggleAreaStatus(array $data)
    {
        $areaManagementStatus = $this->areaManagementStatus->where('area_id', $data['area_id'])->where('branch_id', $data['branch_id'])->first();
        if (!$areaManagementStatus) {
            $areaManagementStatus = AreaManagementStatus::create([
                'area_id' => $data['area_id'],
                'branch_id'=> $data['branch_id'],
                'status' => true,
            ]);
            return $areaManagementStatus;
        }
        $areaManagementStatus->status = !$areaManagementStatus->status;
        $areaManagementStatus->save();
        return $areaManagementStatus;
    }

    /**
     * Get area management status for a specific area and branch
     */
    public function getAreaManagementStatus($areaId, $branchId)
    {
        $areaStatus = $this->areaManagementStatus
            ->where('area_id', $areaId)
            ->where('branch_id', $branchId)
            ->first();

        return [
            'area_id' => $areaId,
            'branch_id' => $branchId,
            'status' => $areaStatus ? $areaStatus->status : true,
            'exists' => $areaStatus !== null
        ];
    }

    public function getPriceByAddress(array $data)
    {
        $now = now();
        $address = $data['address_id'];
        $address = Address::where('id', $address)->first();
        $areaManagement = $this->model->where('area_id', $address->area_id)->where('branch_id', $data['branch_id'])->where('day', $now->format('l'))->where('service_time_from', '<=', $now->format('H:i:s'))->where('service_time_to', '>=', $now->format('H:i:s'))->first();

        // Check if area management status is active
        if ($areaManagement && !$this->isAreaManagementStatusActive($address->area_id, $data['branch_id'])) {
            $areaManagement = null; // Treat as if no area management found
        }

        if (!$areaManagement) {
            Log::info('111');
            // اگر خارج از بازه زمانی یا آدرس بود، بالاترین قیمت ارسال را برگردان
            $highestPriceManagement = $this->model->where('branch_id', $data['branch_id'])
                ->whereNotNull('order_amounts')
                ->get()
                ->map(function ($item) {
                    $orderAmounts = $item->order_amounts;
                    if (is_array($orderAmounts)) {
                        // پیدا کردن بالاترین قیمت orderAmountDelivery از order_amounts
                        $maxPrice = 0;
                        foreach ($orderAmounts as $amount) {
                            if (is_array($amount) && isset($amount['orderAmountDelivery'])) {
                                $deliveryPrice = $amount['orderAmountDelivery'];
                                if (is_numeric($deliveryPrice) && $deliveryPrice > $maxPrice) {
                                    $maxPrice = $deliveryPrice;
                                }
                            }
                        }
                        return [
                            'management' => $item,
                            'max_price' => $maxPrice
                        ];
                    }
                    return null;
                })
                ->filter()
                ->sortByDesc('max_price')
                ->first();
            if ($highestPriceManagement) {
                $price = $highestPriceManagement['max_price'];
                $freeDelivery = $highestPriceManagement['management']->free_delivery;
                $freeDeliveryMinAmount = $highestPriceManagement['management']->free_delivery_min_amount;
            } else {
                return null;
            }
        } else {
            Log::info('222');
            $price = $areaManagement->order_amounts;
            $freeDelivery = $areaManagement->free_delivery;
            $freeDeliveryMinAmount = $areaManagement->free_delivery_min_amount;
        }

        Log::info('333');
        $data = [
            'price' => $price,
            'freeDelivery' => $freeDelivery,
            'freeDeliveryMinAmount' => $freeDeliveryMinAmount,
        ];
        return $data;
    }

    public function getAreaBranches($areaIds)
    {
        $branches = $this->model
            ->with(['branch'])
            ->whereIn('area_id', $areaIds)
            ->select('branch_id')
            ->distinct()
            ->get()
            ->map(function ($areaManagement) {
                return [
                    'id' => $areaManagement->branch->id,
                    'title' => $areaManagement->branch->title,
                    'address' => $areaManagement->branch->address ?? null,
                    'phone' => $areaManagement->branch->phone ?? null,
                    'status' => $areaManagement->branch->status ?? null,
                ];
            });

        return $branches;
    }

    /**
     * Check if branch has any area management records
     */
    public function hasAnyAreaManagement($branchId)
    {
        return $this->model
            ->where('branch_id', $branchId)
            ->exists();
    }
}
