<?php

namespace App\Repositories\AreaSpecialCondition;

use App\Models\AreaSpecialCondition;
use App\Repositories\BaseRepository\BaseRepository;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class EloquentRepository extends BaseRepository implements RepositoryInterface
{
    public function __construct(AreaSpecialCondition $model)
    {
        parent::__construct($model);
    }

    /**
     * Get all special conditions with optional filters
     */
    public function getWithFilter(array $filters = [], bool $paginate = false)
    {
        $query = $this->model->query();

        // Apply filters for JSON fields
        if (isset($filters['branch_id'])) {
            $query->whereJsonContains('branch_id', $filters['branch_id']);
        }

        if (isset($filters['area_id'])) {
            $query->whereJsonContains('area_id', $filters['area_id']);
        }

        if (isset($filters['date'])) {
            $query->where('date', $filters['date']);
        }

        if (isset($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (isset($filters['start_date']) && isset($filters['end_date'])) {
            $query->whereBetween('date', [$filters['start_date'], $filters['end_date']]);
        }

        if (isset($filters['search'])) {
            $search = $filters['search'];
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");

                // Search in branch titles using JSON contains and subquery
                $branchIds = DB::table('branches')
                    ->where('title', 'like', "%{$search}%")
                    ->pluck('id')
                    ->toArray();

                if (!empty($branchIds)) {
                    foreach ($branchIds as $branchId) {
                        $q->orWhereJsonContains('branch_id', $branchId);
                    }
                }

                // Search in area titles using JSON contains and subquery
                $areaIds = DB::table('areas')
                    ->where('title', 'like', "%{$search}%")
                    ->pluck('id')
                    ->toArray();

                if (!empty($areaIds)) {
                    foreach ($areaIds as $areaId) {
                        $q->orWhereJsonContains('area_id', $areaId);
                    }
                }
            });
        }

        // Sorting
        $sortColumn = $filters['sort_column'] ?? 'created_at';
        $sortType = $filters['sort_type'] ?? 'desc';
        $query->orderBy($sortColumn, $sortType);

        if ($paginate) {
            $perPage = $filters['per_page'] ?? 15;
            return $query->paginate($perPage);
        }

        return $query->get();
    }

    /**
     * Get special conditions for a specific branch
     */
    public function getByBranch($branchId, array $filters = [])
    {
        $filters['branch_id'] = $branchId;
        return $this->getWithFilter($filters);
    }

    /**
     * Get special conditions for a specific area
     */
    public function getByArea($areaId, array $filters = [])
    {
        $filters['area_id'] = $areaId;
        return $this->getWithFilter($filters);
    }

    /**
     * Get special conditions for a specific date
     */
    public function getByDate($date, array $filters = [])
    {
        $filters['date'] = $date;
        return $this->getWithFilter($filters);
    }

    /**
     * Get active special conditions within time range
     */
    public function getActiveWithinTimeRange($startTime, $endTime, array $filters = [])
    {
        $query = $this->model->active()
            ->where(function ($q) use ($startTime, $endTime) {
                $q->whereBetween('start_time', [$startTime, $endTime])
                  ->orWhereBetween('end_time', [$startTime, $endTime])
                  ->orWhere(function ($subQ) use ($startTime, $endTime) {
                      $subQ->where('start_time', '<=', $startTime)
                           ->where('end_time', '>=', $endTime);
                  });
            });

        if (isset($filters['branch_id'])) {
            $query->where('branch_id', $filters['branch_id']);
        }

        if (isset($filters['area_id'])) {
            $query->where('area_id', $filters['area_id']);
        }

        if (isset($filters['date'])) {
            $query->where('date', $filters['date']);
        }

        return $query->get();
    }

    /**
     * Check if there are conflicting conditions for the same branch, area, date and time
     */
    public function hasConflictingConditions($branchId, $areaId, $date, $startTime, $endTime, $excludeId = null)
    {
        $query = $this->model->whereJsonContains('branch_id', $branchId)
            ->whereJsonContains('area_id', $areaId)
            ->where('date', $date)
            ->where('status', 'active')
            ->where(function ($q) use ($startTime, $endTime) {
                $q->whereBetween('start_time', [$startTime, $endTime])
                  ->orWhereBetween('end_time', [$startTime, $endTime])
                  ->orWhere(function ($subQ) use ($startTime, $endTime) {
                      $subQ->where('start_time', '<=', $startTime)
                           ->where('end_time', '>=', $endTime);
                  });
            });

        if ($excludeId) {
            $query->where('id', '!=', $excludeId);
        }

        return $query->exists();
    }

    /**
     * Get special conditions that are currently active
     */
    public function getCurrentlyActive(array $filters = [])
    {
        $now = Carbon::now();
        $currentDate = $now->format('Y-m-d');
        $currentTime = $now->format('H:i:s');

        $query = $this->model->active()
            ->where('date', $currentDate)
            ->where('start_time', '<=', $currentTime)
            ->where('end_time', '>=', $currentTime);

        if (isset($filters['branch_id'])) {
            $query->whereJsonContains('branch_id', $filters['branch_id']);
        }

        if (isset($filters['area_id'])) {
            $query->whereJsonContains('area_id', $filters['area_id']);
        }

        return $query->get();
    }
}
