<?php

namespace App\Repositories\Branch;

use App\Models\ActiveTime;
use App\Models\Address;
use App\Models\Branch;
use App\Models\Gallery;
use App\Repositories\BaseRepository\BaseRepository;
use App\Repositories\Category\CategoryRepository;
use App\Repositories\ExportImport\ExportImportRepository;
use App\Repositories\User\UserRepository;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use function Symfony\Component\Translation\t;

class BranchRepository extends BaseRepository implements RepositoryInterface
{

    public function __construct(Branch $model)
    {
        parent::__construct($model);
    }

    public function getActivityTime($branchId)
    {
        $days = [
            'monday' => null, 'tuesday' => null, 'wednesday' => null, 'thursday' => null, 'friday' => null, 'saturday' => null, 'sunday' => null
        ];
        $activities = ActiveTime::where('branch_id', $branchId)->get()->groupBy("week_day")->toArray();
        foreach ($activities as $key => $activity) {
            unset($days[$key]);
        }
        $activities = array_merge($activities, $days);
        return $activities;

    }

    public function setActivityTime($data)
    {
        return ActiveTime::create($data);
    }

    public function destroyActivityTime($branchId, $activityId)
    {
        return ActiveTime::where('id', $activityId)->where("branch_id", $branchId)->delete();
    }

    public function getWithFilter(array $filter = [], $paginate = false)
    {
        try {
            $data = $this->model;

            if (isset($filter['selected']) && !empty($filter['selected'])) {
                $data = $data->select($filter['selected']);
            }

            $data = $data->where(function ($q) use ($filter) {
                if (isset($filter['search']) && !empty($filter['search'])) {
                    $q->where('title', 'like', '%' . $filter['search'] . '%');
                }
                if (isset($filter['category_id']) && !empty($filter['category_id'])) {
                    $categoryChild = app(CategoryRepository::class)->getAllChildArray($filter['category_id']);
                    $q->whereHas('category', function ($query) use ($categoryChild) {
                        $query->whereIn('id', $categoryChild);
                    });
                }

                if (isset($filter['branch_id']) && !empty($filter['branch_id'])) {
                    $q->where('branch_id', $filter['branch_id']);
                }

                if (isset($filter['is_product']) && !empty($filter['is_product']) && $filter['is_product'] == true) {
                    $q->wherehas('products');
                }

                if (isset($filter['agent_id']) && !empty($filter['agent_id'])) {
                    $q->where('agent_id', $filter['agent_id']);
                }

                if (isset($filter['start_date']) && isset($filter['end_date']) && !empty($filter['start_date']) && !empty($filter['end_date'])) {
                    $startDate = \Morilog\Jalali\Jalalian::fromFormat('Y/m/d', $filter['start_date'])->toCarbon();
                    $endDate = \Morilog\Jalali\Jalalian::fromFormat('Y/m/d', $filter['end_date'])->toCarbon();
                    $q->where('created_at', '>=', $startDate)->where('created_at', '<=', $endDate);
                }

                // if(isset($filter['is_admin']) and auth()->check() and auth()->user()->default_address_id!=null){
                //     $address=app(UserRepository::class)->showC(auth()->user()->default_address_id);
                //     $q->whereHas('areaManagements', function ($query) use ($address) {
                //         $query->where('area_id', $address->area_id);
                //     });
                // }else if (!empty($filter['lat']) && !empty($filter['lng'])) {
                //     $lat = $filter['lat'];
                //     $lng = $filter['lng'];
                //     $radius = $filter['radius'] ?? 3; // پیش‌فرض ۱۰ کیلومتر

                //     $q->whereRaw("
                //         (6371 * acos(
                //             cos(radians(?)) * cos(radians(lat)) * cos(radians(lng) - radians(?))
                //             + sin(radians(?)) * sin(radians(lat))
                //         )) <= ?
                //     ", [$lat, $lng, $lat, $radius]);
                //  }

            if (!isset($filter['is_admin'])) {
                    $q->where('status', '!=', 'disable');
                }
            })->with('category:id,title,img,parent_id,branch_id', 'area:id,title', 'user:id,name,family,mobile,target_role_id');


            if (isset($filter['export']) and $filter['export'] == true) {
                $export = clone $data;
                app(ExportImportRepository::class)->createExcelFileBranches($export->get()->toArray());
            }


            if (!empty($filter['sort_column'])) {
                $sortType = (!empty($filter['sort_type'])) ? $filter['sort_type'] : "desc";
                $data = $data->orderBy($filter['sort_column'], $sortType);
            }

            if ($paginate) {
                $countForPa = $filter['count_of_page'] ?? 10;
                return ['status' => 200, "data" => $data->paginate($countForPa), "message" => null];
            }
            if (isset($filter['count_limit'])) {
                $countLimit = (!empty($filter['count_limit'])) ? $filter['count_limit'] : 10;
                return ['status' => 200, "data" => $data->limit($countLimit)->get(), "message" => null];
            }


            return $data->get();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }


    public function search($filter, $paginate = false, $selects = null)
    {
        try {
            $data = $this->model->where(function ($q) use ($filter) {
                if (isset($filter['search']) && !empty($filter['search'])) {
                    $q->where('title', 'like', '%' . $filter['search'] . '%');
                }
            })->where('status', '!=', 'disable')->with('category:id,title,img,parent_id,branch_id', 'area:id,title');

            if ($selects != null)
                $data->select($selects);
            if ($paginate)
                return $data->paginate(25);
            return $data->get();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }


    public function getBranchPhones($branchId)
    {
        return $this->model::find($branchId)->phones;
    }


    public function getGallery($branchId)
    {
        try {
            $branch = Branch::find($branchId);
            return $branch->gallery;
        } catch (\Exception $e) {
            return [];
        }
    }

    public function getGalleryFirst($branchId)
    {
        try {

            $branch = Branch::find($branchId);
            return $branch->gallery()->first();
        } catch (\Exception $e) {
            return [];
        }
    }

    public function createGallery($branchId, $images)
    {
        $new = new Gallery();
        $new->title = " ";
        $new->branch_id = $branchId;
        $new->save();
        return $new->files()->createMany($images);
    }

    public function updateGallery($branchId, $images)
    {
        $gallery = Gallery::where('branch_id', $branchId)->first();
        if ($gallery->files)
            $gallery->files()->delete();
        return $gallery->files()->createMany($images);
    }


    public function getNowStatus($branchId)
    {
        $dayName = Carbon::now()->format('l');
        $nowTime = Carbon::now()->format('H:i:s');

        $holiday = \App\Models\BranchHoliday::active()
            ->forBranch($branchId)
            ->forDate(Carbon::now())
            ->get()
            ->first(function ($holiday) use ($nowTime) {
                return $holiday->isTimeInHoliday(Carbon::now(), $nowTime);
            });

        if ($holiday) {
            return null;
        }

        return ActiveTime::where('branch_id', $branchId)->where('week_day', $dayName)
            ->where('from_time', '<=', $nowTime)->
            where('to_time', '>=', $nowTime)->first();
    }

    public function checkClockActivity($branchId, $weekDay, $time)
    {

        if ($weekDay == null)
            $weekDay = Carbon::now()->format('l');
        if ($time == null)
            $time = Carbon::now()->format('H:i:s');

        return ActiveTime::where('branch_id', $branchId)->where('week_day', $weekDay)
            ->where('from_time', '<=', $time)->
            where('to_time', '>=', $time)->first();
    }


    public function perOrderTime($branchId)
    {
        $branch = Branch::find($branchId);
        if ($branch) {
            if (!$branch->per_order_status)
                return ['message' => 'پیش سفارش رستوران فعال نیست', 'status' => 400, 'data' => []];

            $day = $branch->per_order_day;

            if ($day > 6) $day = 6;
            $dataArray = [];
            $days = [];

            $now = Carbon::now();
            $dayName = $now->copy()->format('l');
            $nowTime = $now->copy()->format('H:i:s');
            $data = ActiveTime::where('branch_id', $branch->id)->where('week_day', $dayName)
                ->where('from_time', '<=', $nowTime)->
                where('to_time', '>=', $nowTime)->get();


            foreach ($data as $item) {
                $dataArray[$dayName] = array_merge($dataArray[$dayName] ?? [], $this->sortArray_time($item));
            }

            for ($i = $day; $i >= 1; $i--) {
                $days[] = $now->copy()->addDays($i)->format('l');
            }

            $data = ActiveTime::where('branch_id', $branch->id)->whereIn('week_day', $days)->get();
            foreach ($data as $item) {
                $dataArray[$item->week_day] = array_merge($data[$item->week_day] ?? [], $this->sortArray_time($item));
            }
            return ['data' => $dataArray, 'message' => '', 'status' => 200];
        }
        return ['message' => 'پیش سفارش رستوران فعال نیست', 'status' => 400, 'data' => []];
    }


    public function sortArray_time($data)
    {
        $intervals = [];
        $startTime = new \DateTime($data->from_time); // تبدیل به DateTime
        $endTime = new \DateTime($data->to_time); // تبدیل به DateTime
        while ($startTime <= $endTime) {
            $intervals[] = $startTime->format('H:i'); // فرمت زمان مورد نظر
            $startTime->add(new \DateInterval('PT30M')); // اضافه کردن ۳۰ دقیقه
        }
        return array_unique($intervals);
    }


    public function getActivityDay($branchId)
    {
        $dayName = Carbon::now()->format('l');
        return ActiveTime::where('branch_id', $branchId)->where('week_day', $dayName)->first();

    }

    public function showC($id)
    {
        try {
            return $this->model->where('id', $id)->where('status', '!=', 'disable')
                ->with('area:id,title,parent_id,lat,lng', 'otherBranch:id,title,vendor_id', 'nearestOpeningHour:id,from_time,to_time,week_day,branch_id', 'vendor')->first();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }

    public function setBranchArea(array $data): array
    {
        $branch = $this->show($data['branch_id']);
        foreach ($data['areas'] as $area) {
            $branch->deliveryAreas()->syncWithoutDetaching([
                (int)$area['id'] => ['shipment_price' => $area['shipment_price']]
            ]);
        }
        return ['status' => 200, 'message' => 'success', 'data' => []];
    }

    public function detachBranchArea(array $data): array
    {
        $branch = $this->show($data['branch_id']);
        $branch->deliveryAreas()->detach([$data['area_id']]);
        return ['status' => 200, 'message' => 'success', 'data' => []];
    }

    public function setBranchDeliveryTypes($branchId, array $data): array
    {
        try {
            Log::error($branchId);
            $branch = $this->show($branchId);
            $data = $branch->deliveryTypes()->sync($data);
            return ['status' => 200, 'message' => 'success', 'data' => $data];
        } catch (\Exception $e) {
            handelError('error', $e);
            return ['status' => 500, 'message' => 'error server', 'data' => []];
        }
    }

    public function getBranchDeliveryTypes($branchId): array
    {
        try {
            $branch = $this->show($branchId);
            $data = $branch->deliveryTypes()->pluck('id')->toArray() ?? [];
            return ['status' => 200, 'message' => 'success', 'data' => $data];
        } catch (\Exception $e) {
            handelError('error', $e);
            return ['status' => 500, 'message' => 'error server', 'data' => []];
        }
    }

    public function getBranchDeliveryAreas($data)
    {
        $branch = Branch::find($data['branch_id']);
        $data = $branch->deliveryAreas()->select('areas.id', 'areas.title', 'shipment_price')->get()->toArray() ?? null;
        return ['status' => 200, 'message' => 'success', 'data' => $data];
    }


    public function sitemapList()
    {
        try {
            return $this->model->select('id', 'title')->get()->toArray();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }

    public function getAllBracnhMainPluck($select = ['id', 'title'])
    {
        return $this->model->select($select)->pluck('title', 'id')->toArray();
    }
}
