<?php

namespace App\Repositories\BranchHoliday;

use App\Enums\HolidayReasonType;
use App\Models\BranchHoliday;
use App\Repositories\BaseRepository\BaseRepository;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class BranchHolidayRepository extends BaseRepository implements RepositoryInterface
{
    public function __construct(BranchHoliday $model)
    {
        parent::__construct($model);
    }

    /**
     * create holiday
     */
    public function createHoliday(array $data)
    {
        try {
            // validate data
            $this->validateHolidayData($data);

            return $this->model->create($data);
        } catch (\Exception $e) {
            Log::error('Error creating holiday: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * update holiday
     */
    public function updateHoliday($id, array $data)
    {
        try {
            $holiday = $this->model->findOrFail($id);
            $this->validateHolidayData($data);

            $holiday->update($data);
            return $holiday;
        } catch (\Exception $e) {
            Log::error('Error updating holiday: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * delete holiday
     */
    public function deleteHoliday($id)
    {
        try {
            $holiday = $this->model->findOrFail($id);
            return $holiday->delete();
        } catch (\Exception $e) {
            Log::error('Error deleting holiday: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * get holidays
     */
    public function getBranchHolidays($filters = [], $branchId)
    {
        $query = $this->model->where('branch_id', $branchId);
        if (isset($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (isset($filters['holiday_type'])) {
            $query->where('holiday_type', $filters['holiday_type']);
        }

        if (isset($filters['start_date'])) {
            $query->where('start_date', '>=', $filters['start_date']);
        }

        if (isset($filters['end_date'])) {
            $query->where('end_date', '<=', $filters['end_date']);
        }

        if (isset($filters['holiday_reason_type'])) {
            $query->where('holiday_reason_type', $filters['holiday_reason_type']);
        }

        return $query->get();
    }

    /**
     * check holiday
     */
    public function checkHoliday($date, $time = null, $branchId)
    {
        $checkDate = $date instanceof Carbon ? $date : Carbon::parse($date);

        $holidays = $this->model
            ->active()
            ->forBranch($branchId)
            ->forDate($checkDate)
            ->get();

        foreach ($holidays as $holiday) {
            if ($holiday->isTimeInHoliday($checkDate, $time)) {
                return $holiday;
            }
        }

        return null;
    }

    /**
     * get active holidays in range
     */
    public function getActiveHolidaysInRange($startDate, $endDate, $branchId)
    {
        return $this->model
            ->active()
            ->forBranch($branchId)
            ->where(function ($query) use ($startDate, $endDate) {
                $query->where(function ($q) use ($startDate, $endDate) {
                    $q->where('start_date', '<=', $endDate)
                      ->where(function ($subQ) use ($startDate) {
                          $subQ->whereNull('end_date')
                                ->orWhere('end_date', '>=', $startDate);
                      });
                });
            })
            ->orderBy('start_date')
            ->get();
    }

    /**
     * validate holiday data
     */
    private function validateHolidayData(array $data)
    {
        $rules = [
            'branch_id' => 'required|exists:branches,id',
            'holiday_type' => 'nullable|in:single_date,date_range,weekly_pattern',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'start_time' => 'nullable|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i|after:start_time',
            'is_full_day' => 'nullable|boolean',
            'week_days' => 'nullable|array',
            'week_days.*' => 'string|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'repeat_yearly' => 'nullable|boolean',
            'reason' => 'nullable|string|max:255',
            'holiday_reason_type' => 'nullable|in:' . implode(',', array_column(HolidayReasonType::cases(), 'value')),
            'status' => 'nullable|in:active,inactive'
        ];

        $validator = validator($data, $rules);

        if ($validator->fails()) {
            throw new \InvalidArgumentException($validator->errors()->first());
        }

        // validate extra data based on holiday type (only if holiday_type is provided)
        if (isset($data['holiday_type']) && !empty($data['holiday_type'])) {
            switch ($data['holiday_type']) {
                case 'single_date':
                    if (isset($data['end_date'])) {
                        throw new \InvalidArgumentException('برای تعطیلی تک روزه، end_date نباید تنظیم شود');
                    }
                    break;

                case 'date_range':
                    if (!isset($data['end_date'])) {
                        throw new \InvalidArgumentException('برای تعطیلی بازه‌ای، end_date الزامی است');
                    }
                    break;

                case 'weekly_pattern':
                    if (!isset($data['week_days']) || empty($data['week_days'])) {
                        throw new \InvalidArgumentException('برای تعطیلی هفتگی، week_days الزامی است');
                    }
                    break;
            }
        }
    }

    /**
     * get upcoming holidays
     */
    public function getUpcomingHolidays($limit = 10, $branchId)
    {
        return $this->model
            ->active()
            ->forBranch($branchId)
            ->where('start_date', '>=', Carbon::today())
            ->orderBy('start_date')
            ->limit($limit)
            ->get();
    }

    /**
     * check branch status
     */
    public function isBranchOpen($date = null, $time = null, $branchId)
    {
        $checkDate = $date ? Carbon::parse($date) : Carbon::now();
        $checkTime = $time ? Carbon::parse($time) : Carbon::now();

        // check holiday
        $holiday = $this->checkHoliday($checkDate, $checkTime->format('H:i:s'), $branchId);
        if ($holiday) {
            return false;
        }

        // check active time
        $dayName = strtolower($checkDate->format('l'));
        $timeStr = $checkTime->format('H:i:s');

        $activeTime = \App\Models\ActiveTime::where('branch_id', $branchId)
            ->where('week_day', $dayName)
            ->where('from_time', '<=', $timeStr)
            ->where('to_time', '>=', $timeStr)
            ->first();

        return $activeTime !== null;
    }
}
