<?php

namespace App\Repositories\Category;

use App\Models\Category;
use App\Repositories\BaseRepository\BaseRepository;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

abstract class EloquentRepository extends BaseRepository
{
    public function __construct(Category $model)
    {
        parent::__construct($model);
    }

    public function getWithFilter(array $filter = [])
    {
        try {
            $query = $this->model->newQuery();

            if (!empty($filter['search'])) {
                $query->where('title', 'like', '%' . $filter['search'] . '%');
            }

            if (array_key_exists('parent_id', $filter) && $filter['parent_id'] !== '' && $filter['parent_id'] !== null) {
                $query->where('parent_id', $filter['parent_id']);
            } else {
                $query->where('parent_id', 0);
            }

            if (array_key_exists('branch_id', $filter) && $filter['branch_id'] !== '' && $filter['branch_id'] !== null) {
                $query->where('branch_id', $filter['branch_id']);
            } else {
                $query->where('branch_id', 0);
            }

            if (!empty($filter['time'])) {
                $date = $filter['time'];
                $query->where(function ($q) use ($date) {
                    $q->whereNull('start_serving')
                        ->orWhereNull('end_serving')
                        ->orWhere(function ($q) use ($date) {
                            $q->where('start_serving', '<=', $date)->where('end_serving', '>=', $date);
                        });
                });
            }

            if (!empty($filter['select'])) {
                $query->select($filter['select']);
            }

            $query->withCount('children');

            return $query->get();
        } catch (\Throwable $e) {
            Log::error($e->getMessage());
            return null;
        }
    }

    public function getCategoryAllWithProduct($filter)
    {
        try {
            $data = $this->model
                ->where('status', 'active')
                ->where(function ($q) use ($filter) {
                    if (isset($filter['branch_id']) && !empty($filter['branch_id'])) {
                        $q->where('branch_id', $filter['branch_id']);
                    }
                })
                ->with('productMenu:id,branch_id,menu_id,image,title,description,rate,status')
                ->whereHas('productMenu', function ($query) use ($filter) {
                    if (isset($filter['title']) && $filter['title'] !== null && $filter['title'] != ' ') {
                        $query->whereLike('title', $filter['title']);
                    }
                })
                ->where(function ($query) use ($filter) {
                    if (isset($filter['time']) and !empty($filter['time'])) {
                        $date = $filter['time'];
                    } else {
                        $date = Carbon::now()->format('H:i:s');
                    }

                    $query
                        ->whereNull('start_serving')
                        ->orWhereNull('end_serving')
                        ->orWhere(function ($query) use ($date) {
                            $query->where('start_serving', '<=', $date)->where('end_serving', '>=', $date);
                        });
                });
            if (isset($filter['select']) and !empty($filter['select'])) {
                return $data->select($filter['select'])->get();
            }

            return $data->get();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return [];
        }
    }

    public function getAllChildArray($ids)
    {
        if ($ids instanceof \Illuminate\Support\Collection) {
            $ids = $ids->all();
        } else {
            $ids = (array) $ids;
        }

        $categoryIds = $this->model->whereIn('parent_id', $ids)->where('status', 'active')->pluck('id')->toArray();

        $result = array_merge($ids, $categoryIds);

        $result = array_values(array_unique($result));

        return $result;
    }

    public function getAllCategoryMainPluck($select = ['id', 'title', 'parent_id', 'branch_id'])
    {
        return $this->model->where('branch_id', 0)->where('status', 'active')->whereNull('deleted_at')->select($select)->pluck('title', 'id')->toArray();
    }
    
    public function toggleStatus($id)
    {
        try {
            $category = $this->model->find($id);
            if (!$category) {
                return null;
            }

            $newStatus = $category->status === 'active' ? 'inactive' : 'active';
            $category->status = $newStatus;
            $category->save();

            return $category;
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }
}
