<?php

namespace App\Repositories\Comment;

use App\Models\Comment;
use App\Repositories\BaseRepository\BaseRepository;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

abstract class  EloquentRepository extends BaseRepository
{
    abstract public function model();

    public function __construct(Comment $model)
    {
        parent::__construct($model);
    }

    public function getWithFilter($filter, $paginate = true)
    {
        try {
            $data = $this->model->where(function ($q) use ($filter) {

                if (isset($filter['user_id']) && !empty($filter['user_id'])) {
                    $q->where('user_id', $filter['user_id']);
                }

                if (isset($filter['branch_id']) && !empty($filter['branch_id'])) {
                    $q->where('branch_id', $filter['branch_id']);
                }

                if (isset($filter['start_date']) && isset($filter['end_date']) && !empty($filter['start_date']) && !empty($filter['end_date'])) {
                    $startDate = \Morilog\Jalali\Jalalian::fromFormat('Y/m/d', $filter['start_date'])->toCarbon();
                    $endDate = \Morilog\Jalali\Jalalian::fromFormat('Y/m/d', $filter['end_date'])->toCarbon();
                    $q->where('created_at', '>=', $startDate)->where('created_at', '<=', $endDate);
                }

            })->where('parent_id', 0)->with('child', 'user:name,family,id', 'branch:id,title');
            if ($paginate)
                return $data->paginate(25);
            return $data->get();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }

    public function showA($id)
    {
        return $this->model->where('id', $id)->where('parent_id', 0)->with('child', 'user:name,family,id', 'branch:id,title')->first();
    }

    public function getCommentProduct($productId)
    {
        $this->model->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(products, '$[*].id')) = ?", [$productId])
            ->where('parent_id', 0)
            ->with('user:id,name,family', 'child')
            ->get();
    }


    public function getCommentBranch($branchId)
    {
        return $this->model->where('branch_id', $branchId)->where('parent_id', 0)->with('child')->get();
    }

    public function getBranchRate($branchId)
    {
        $sumAndCount = $this->model->where('branch_id', $branchId)->where('parent_id', 0)->select(DB::raw('SUM(rate) as total_rate'), DB::raw('COUNT(*) as total_count'))
            ->first();
        if($sumAndCount->total_rate==0 || $sumAndCount->total_count ==0)
            return 0;
        return $sumAndCount->total_rate / $sumAndCount->total_count ?? 0;
    }

    public function checkCommentUserOrder($orderId)
    {
        return $this->model->where('order_id', $orderId)->where('parent_id', 0)->where('user_id', auth()->id())->first();
    }

    /**
     * پیدا کردن نظر بر اساس ID
     */
    public function find($data = [])
    {
        try {
            if (isset($data['comment_id'])) {
                return $this->model->where('id', $data['comment_id'])
                    ->with('child', 'user:name,family,id', 'branch:id,title')
                    ->first();
            }
            return null;
        } catch (\Exception $e) {
            Log::error('Error in find: ' . $e->getMessage());
            return null;
        }
    }


    public function toggleApproval($commentId)
    {
        try {
            $comment = $this->model->find($commentId);
            if (!$comment) {
                return false;
            }

            $comment->admin_approve = !$comment->admin_approve;
            $comment->save();

            return $comment;
        } catch (\Exception $e) {
            Log::error('Error in toggleApproval: ' . $e->getMessage());
            return false;
        }
    }
}
