<?php

namespace App\Repositories\NotificationMobile;

use App\Enums\NotificationType;
use App\Models\NotificationMobile;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Pagination\LengthAwarePaginator;

class NotificationMobileRepository implements NotificationMobileRepositoryInterface
{
    public function getAll(array $filters = []): Collection
    {
        $query = NotificationMobile::with('branch');

        if (isset($filters['branch_id'])) {
            $query->where('branch_id', $filters['branch_id']);
        }

        if (isset($filters['type'])) {
            $query->where('type', $filters['type']);
        }

        if (isset($filters['is_active'])) {
            $query->where('is_active', $filters['is_active']);
        }

        if (isset($filters['mobile'])) {
            $query->where('mobile', 'like', '%' . $filters['mobile'] . '%');
        }

        return $query->orderBy('created_at', 'desc')->get();
    }

    public function getPaginated(array $filters = [], int $perPage = 15): LengthAwarePaginator
    {
        $query = NotificationMobile::with('branch');

        if (isset($filters['branch_id'])) {
            $query->where('branch_id', $filters['branch_id']);
        }

        if (isset($filters['type'])) {
            $query->where('type', $filters['type']);
        }

        if (isset($filters['is_active'])) {
            $query->where('is_active', $filters['is_active']);
        }

        if (isset($filters['mobile'])) {
            $query->where('mobile', 'like', '%' . $filters['mobile'] . '%');
        }

        return $query->orderBy('created_at', 'desc')->paginate($perPage);
    }

    public function findById(int $id): ?NotificationMobile
    {
        return NotificationMobile::with('branch')->find($id);
    }

    public function create(array $data): NotificationMobile
    {
        return NotificationMobile::create($data);
    }

    public function update(int $id, array $data): bool
    {
        $notificationMobile = NotificationMobile::find($id);

        if (!$notificationMobile) {
            return false;
        }

        return $notificationMobile->update($data);
    }

    public function delete(int $id): bool
    {
        $notificationMobile = NotificationMobile::find($id);

        if (!$notificationMobile) {
            return false;
        }

        return $notificationMobile->delete();
    }

    public function getByBranch(int $branchId): Collection
    {
        return NotificationMobile::with('branch')
            ->where('branch_id', $branchId)
            ->orderBy('type')
            ->get();
    }

    public function getByType(NotificationType $type): Collection
    {
        return NotificationMobile::with('branch')
            ->where('type', $type)
            ->orderBy('branch_id')
            ->get();
    }

    public function getByBranchAndType(int $branchId, NotificationType $type): Collection
    {
        return NotificationMobile::with('branch')
            ->where('branch_id', $branchId)
            ->where('type', $type)
            ->get();
    }

    public function getMobileNumbers(int $branchId, NotificationType $type): array
    {
        return NotificationMobile::where('branch_id', $branchId)
            ->where('type', $type)
            ->pluck('mobile')
            ->toArray();
    }

    public function getActiveMobileNumbers(int $branchId, NotificationType $type): array
    {
        return NotificationMobile::where('branch_id', $branchId)
            ->where('type', $type)
            ->where('is_active', true)
            ->pluck('mobile')
            ->toArray();
    }
}
