<?php

namespace App\Repositories\Order;

use App\Enums\OrderStatus;
use App\Enums\PaymentStatus;
use App\Enums\TransactionType;
use App\Events\OrderChangeStatus;
use App\Models\Address;
use App\Models\Coupon;
use App\Models\Extra;
use App\Models\Gateway;
use App\Models\Order;
use App\Models\OrderCart;
use App\Models\Setting;
use App\Models\User;
use App\Models\UserBranchWallet;
use App\Models\Variety;
use App\Repositories\BaseRepository\BaseRepository;
use App\Repositories\Branch\BranchRepository;
use App\Repositories\Comment\CommentRepository;
use App\Repositories\Delivery\DeliveryRepository;
use App\Repositories\ExportImport\ExportImportRepository;
use App\Repositories\Address\AddressRepository;
use App\Repositories\Setting\SettingRepository;
use App\Repositories\Transaction\TransactionRepository;
use App\Repositories\User\UserRepository;
use App\Repositories\Variety\VarietyRepository;
use Carbon\Carbon;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Morilog\Jalali\Jalalian;

abstract class  EloquentRepository extends BaseRepository
{
    abstract public function model();

    abstract public function modelOrderCart();

    private $guest_token = null;

    protected $addressRepository;

    public function __construct(Order $model)
    {
        parent::__construct($model);
        $this->addressRepository = app(AddressRepository::class);
    }

    public function showC($id)
    {
        try {
            return $this->model->where('id', $id)->with('user', 'deliveryType:id,title')->first();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }


    public function showA($id)
    {
        try {
            return $this->model->where('id', $id)->with('customer','branch:id,title,address,lat,lng,logo','user','driver', 'orderCart','deliveryType:id,title')->first();
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return null;
        }
    }
    public function showWithUser($id)
    {
        try {
            return $this->model->where('id', $id)->where('user_id',auth()->id())
                ->with([
                    'user:id,name,family,mobile,email',
                    'orderCart',
                    'branch:id,title,address,logo,min_order,vendor_id,lat,lng,cash_back_status,cash_back_percent,cash_back_max_price',
                    'address_order:id,user_id,address,plaque,postal_code,unit,lat,lng',
                    'deliveryType'
                ])->first();
        } catch (\Exception $e) {
            Log::info($e->getMessage());
            return null;
        }
    }

    public function getWithFilter(array $filter = [], $paginate = true)
    {
        try {
            $data = $this->model->where(function ($q) use ($filter) {
                if (isset($filter['user_id']) && !empty($filter['user_id'])) {
                    $q->whereHas('user', function ($q1) use ($filter) {
                        $q1->where('user_id', $filter['user_id']);
                    });
                }
                if (isset($filter['branch_id']) && !empty($filter['branch_id'])) {
                    $q->whereHas('branch', function ($q1) use ($filter) {
                        $q1->where('id', $filter['branch_id']);
                    });
                }
                if (isset($filter['payment_status']) && !empty($filter['payment_status'])) {
                    $q->where('payment_status', $filter['payment_status']);
                }
                if (filled($filter['order_status'] ?? null)) {
                    $value = $filter['order_status'];

                    $statuses = is_array($value)
                        ? $value
                        : (strpos($value, ',') !== false
                            ? array_map('trim', explode(',', $value))
                            : [$value]);

                    $q->whereIn('order_status', $statuses);
                }

                if (isset($filter['type_order']) && !empty($filter['type_order'])) {
                    $q->where('type_order', $filter['type_order']);
                }
                if (isset($filter['user_id']) && !empty($filter['user_id'])) {
                    $q->where('user_id', $filter['user_id']);
                }
                if (isset($filter['delivery_time']) && !empty($filter['delivery_time'])) {
                    $q->where('delivery_time', '>=', $filter['delivery_time'])->andwhere('delivery_time', '<=', $filter['delivery_time']);
                }
                if (isset($filter['payment_type']) && !empty($filter['payment_type'])) {
                    $q->where('payment_type', $filter['payment_type']);
                }
                if (isset($filter['driver_id']) && !empty($filter['driver_id'])) {
                    $q->where('driver_id', $filter['driver_id']);
                }
                if (isset($filter['payment_getway_id']) && !empty($filter['payment_getway_id'])) {
                    $q->where('payment_getway_id', $filter['payment_getway_id']);
                }
                if (isset($filter['agent_id']) && !empty($filter['agent_id'])) {
                    $q->whereHas('branch', function ($query) use ($filter) {
                        $query->where('agent_id', $filter['agent_id']);
                    });
                }
                if (isset($filter['start_date']) && isset($filter['end_date']) && !empty($filter['start_date']) && !empty($filter['end_date'])) {
                    $startDate = \Morilog\Jalali\Jalalian::fromFormat('Y/m/d', $filter['start_date'])->toCarbon();
                    $endDate = \Morilog\Jalali\Jalalian::fromFormat('Y/m/d', $filter['end_date'])->toCarbon();
                    $q->where('created_at', '>=', $startDate)->where('created_at', '<=', $endDate);
                }
            })->where('order_status', '!=', 'wating_payment')->where('order_status', '!=', 'ordering')->orderBy('updated_at', 'desc')->with('user:id,name,family,mobile,email', 'branch:id,title,address,logo,min_order,vendor_id');
            $countAll = clone $data;
            $sumPriceAll = clone $data;
            $sumDelivery = clone $data;
            $sumTax = clone $data;
            $sumDiscount = clone $data;
            $latestMonthData = clone $data;
            if (isset($filter['export']) and $filter['export'] == true) {
                $export = clone $data;
                app(ExportImportRepository::class)->createExcelFileOrders($export->get()->toArray());
            }
            $latestMonth = Carbon::now()->format('Y-m'); // Get the current month in YYYY-MM format
            $dayCounts = []; // Initialize an array to store the count for each day
            $transactions = $latestMonthData->where('created_at', '>=', Carbon::now()->subMonth())->select('id', 'created_at')
                ->get();
            foreach ($transactions as $transaction) {
                $date = Carbon::parse($transaction->created_at);
                $day = $date->format('Y-m-d');
                if (!isset($dayCounts[$day])) {
                    $dayCounts[$day] = 0;
                }
                $dayCounts[$day]++;
            }
            if ($dayCounts)
                ksort($dayCounts);
            $latestMonthData = $dayCounts;
            if ($paginate)
                return ["data" => $data->paginate(25), 'countAll' => $countAll->count(), "sumPriceAll" => $sumPriceAll->sum('order_price')
                    , "sumDelivery" => $sumDelivery->sum('delivery_price'), 'sumTax' => $sumTax->sum('tax_price'), 'sumDiscount' => $sumDiscount->sum('discount_price')
                    , "latestMonthData" => $latestMonthData,
                ];
            return ["data" => $data->get(), 'countAll' => $countAll->count(), "sumPriceAll" => $sumPriceAll->sum('order_price')
                , "sumDelivery" => $sumDelivery->sum('delivery_price'), 'sumTax' => $sumTax->sum('tax_price'), 'sumDiscount' => $sumDiscount->sum('discount_price'), "latestMonthData" => $latestMonthData
            ];
        } catch (\Exception $e) {
            Log::error($e);
            return null;
        }
    }


    public function getLatestUserRegisterCount()
    {

        $users = User::where('created_at', '>=', Carbon::now()->subMonth())->select('id', 'created_at')
            ->get();

        $dayCounts = [];
        foreach ($users as $user) {
            $date = Carbon::parse($user->created_at);
            $day = $date->format('Y-m-d');
            if (!isset($dayCounts[$day])) {
                $dayCounts[$day] = 0;
            }
            $dayCounts[$day]++;
        }
        if ($dayCounts)
            ksort($dayCounts);
        return $dayCounts;
    }

    public function changeStatus($orderId, $status)
    {
        try {
            DB::beginTransaction();
            $order = $this->model->where("id", $orderId)->first();
            $branch = app(BranchRepository::class)->show($order->branch_id);

            if ($order->order_status != $status) {
                if ($status == "delivered") {

                    if($branch->percent_order == null){
                        $percentageSales = app(SettingRepository::class)->getSettingWithName('percentageSales')['value'];
                    }
                    else{
                        $percentageSales = $branch->percent_order;
                    }
                    $percentageDeliver = app(SettingRepository::class)->getSettingWithName('percentageDeliver')['value'];
                    if ($order->driver_id != null) {
                        //deliver
                        $deliverPrice = PriceCalculation($order->delivery_price, $percentageDeliver);
                        $deliverProfit = $order->delivery_price - $deliverPrice;
                        app(TransactionRepository::class)->storeP(['amount' => $order->delivery_price, 'target_id' => $orderId, 'user_id' => $order->driver_id, 'transaction_type' => TransactionType::DELIVER, 'description' => 'هزینه بابت سفارش شماره : ' . $orderId]);
                        app(TransactionRepository::class)->storeP(['amount' => $deliverProfit * -1, 'target_id' => $orderId, 'user_id' => $order->driver_id, 'transaction_type' => TransactionType::DELIVER, 'description' => 'سود هزینه ارسال سفارش : ' . $orderId]);

                    }
                    //order
                    $orderPriceM = $order->order_price - $order->delivery_price;

                    $orderPrice = PriceCalculation($orderPriceM, $percentageSales);
                    $orderProfit = $orderPriceM - $orderPrice;
                    $managementVendor = User::where('target_role_id', $order->branch_id)->first();
                    app(TransactionRepository::class)->storeP(['amount' => $orderPriceM, 'target_id' => $orderId, 'user_id' => $managementVendor->id, 'transaction_type' => TransactionType::SALES, 'description' => 'فروش سفارش  : ' . $orderId]);
                    app(TransactionRepository::class)->storeP(['amount' => $orderProfit * -1, 'target_id' => $orderId, 'user_id' => $managementVendor->id, 'transaction_type' => TransactionType::SALESPROFIT, 'description' => 'سود از  سفارش : ' . $orderId]);

                    if ($branch->cash_back_status and $branch->cash_back_percent != 0) {
                        $orderPrice = $orderPriceM;
                        $pCashBack = $this->cashBackCalculation($orderPrice, $branch->cash_back_percent, $branch->cash_back_max_price);
                        $expiredAt = null;
                        if (!empty($branch->cash_back_time) && (int)$branch->cash_back_time > 0) {
                            $expiredAt = Carbon::now()->addDays((int)$branch->cash_back_time);
                        }
                        app(UserRepository::class)->setWalletBranch($branch->id, $order->user_id, $pCashBack, $expiredAt);
                        app(SMSRepositoryInterface::class)->sendSMSCashBack($order->user->mobile, $branch->title, $pCashBack, $order->created_at);
                    }
                }
                $order->update(['order_status' => $status]);
                event(new OrderChangeStatus($order));
                DB::commit();
            }
            return ['data' => [], 'message' => 'با موفقیت انجام شد', 'status' => 200];
        } catch (\Exception $e) {
            DB::rollback();
            Log::error($e);
            return ['data' => [], 'message' => $e->getMessage(), 'order_status' => $status, 'status' => 500];
        }
    }

    private function cashBackCalculation($orderPrice, $cashBackPercent, $cashBackMaxPrice)
    {
        $cashBack = $orderPrice - PriceCalculation($orderPrice, $cashBackPercent);
        if ($cashBack > $cashBackMaxPrice)
            $cashBack = $cashBackMaxPrice;
        return $cashBack;
    }

    public function acceptOrder($orderId, $minute)
    {
        try {
            DB::beginTransaction();
            $order = $this->model->where("id", $orderId)->first();
            $currentDateTime = Carbon::now();
            $preparationTime = $currentDateTime->addMinutes($minute);
            $update = $order->update(['order_status' => 'preparing', 'preparation_time' => $preparationTime]);
            DB::commit();
            event(new OrderChangeStatus($order));
            return ['data' => $update, 'status' => 200, 'message' => null];
        } catch (\Exception $e) {
            DB::rollback();
            handelError('error', $e->getMessage());
            return ['data' => [], 'status' => 500, 'message' => null];

        }
    }


    public function addCartOrder($data)
    {
        DB::beginTransaction();
        try {
            if (isset($data['guest_token']) and $data['guest_token'] != null)
                $this->guest_token = $data['guest_token'];

            $deliveryType = null;
            if (isset($data['delivery_type_id'])) {
                $deliveryType = \App\Models\DeliveryType::find($data['delivery_type_id']);
            }
            if (!$deliveryType || !in_array($deliveryType->delivery_method, ['serve', 'actual'])) {
                $area_id = null;
                // فقط در صورت وجود آدرس، بررسی منطقه انجام شود
                if (isset($data['address_id']) && $data['address_id'] != null && $data['address_id'] != "null") {
                    $address = \App\Models\Address::find($data['address_id']);

                    if ($address && $address->lat && $address->lng) {
                        $areas = \App\Models\Area::all();
                        $point = [$address->lat, $address->lng];
                        foreach ($areas as $area) {
                            if (is_array($area->coordinates) && \App\Helpers\PointInPolygonHelper::check($point, $area->coordinates)) {
                                $area_id = $area->id;
                                break;
                            }
                        }
                        if (!$area_id) {
                            return ['status' => 400, 'message' => 'آدرس شما در هیچ منطقه‌ای قرار ندارد.', 'data' => []];
                        }
                    } else {
                        return ['status' => 400, 'message' => 'مختصات آدرس نامعتبر است.', 'data' => []];
                    }
                }
                // اگر آدرس وجود نداشت، area_id همچنان null باقی می‌ماند
            } else {
                $area_id = null;
            }

            $orderActive = $this->getLatestOrderUser();
            if (isset($data['extraIds']) and $data['extraIds'] != "") {
                $extras = $this->getExtras($data['extraIds']);
                $varietyCart = $this->checkProductOrder($orderActive, $data['variety_id'], $extras->pluck('id')->implode('-') ?? null);
            } else
                $varietyCart = $this->checkProductOrder($orderActive, $data['variety_id'], null);
            $varietyRep = app(VarietyRepository::class);
            $variety = $varietyRep->getById($data['variety_id']);
            $branch = $varietyRep->getBranch($data['variety_id']) ?? null;
            if (!$branch)
                return ['status' => 400, 'message' => 'فروشنده نا معتبر است.', 'data' => []];

            if ($orderActive->branch_id == null)
                $orderActive->update(['branch_id' => $branch->id]);
            elseif ($orderActive->branch_id != $branch->id)
                return ['status' => 400, 'message' => 'شما سبد خرید باز دیگری دارید.', 'data' => ['order_active' => true]];

            if (!$deliveryType || !in_array($deliveryType->delivery_method, ['serve', 'actual'])) {
                // فقط در صورت وجود area_id، بررسی ساعت فعالیت انجام شود
                if ($area_id != null) {
                    $statusClock = app(\App\Repositories\AreaManagment\AreaManagmentRepository::class)
                        ->checkClockActivity($area_id, $branch->id, $data['week_day'] ?? null, $data['time'] ?? null);
                    if ($statusClock == null)
                        return ['status' => 400, 'message' => 'در این زمان فروشگاه منطقه انتخابی بسته است.', 'data' => []];
                }
            }
            $date = $this->getNextDateForDay($data['week_day'], $data['time']);
            $this->update($orderActive->id, ["delivery_time" => $date]);

            if ($variety->count == 0)
                return ['status' => 400, 'message' => 'موجودی محصول تمام شده است', 'data' => []];

            if ($variety->count < $data['count'])
                return ['status' => 400, 'message' => 'تعداد درخواستی بیشتر از موجودی می باشد', 'data' => []];

            if (!$varietyCart) {
                $cart = OrderCart::create([
                    'order_id' => $orderActive->id,
                    'variety_id' => $data['variety_id'],
                    'count' => $data['count'],
                    'extras' => (isset($extras)) ? json_encode($extras) : null,
                    'extra_ids' => (isset($extras)) ? $extras->pluck('id')->implode('-') : null,
                    'price' => $variety->price,
                    'discount' => $variety->discount
                ]);
            } else {
                $varietyCart->update(['count' => $varietyCart->count + $data['count']]);
            }
            if (auth()->check()) {
                $check = app(VarietyRepository::class)->decrementCount($variety->id, $data['count']);
                if (!$check) {
                    DB::rollBack();
                    return ['status' => 400, 'message' => 'موجودی محصول تمام شده است', 'data' => []];
                }
            }
            DB::commit();
            return ['status' => 200, 'message' => 'محصول با موفقیت به سبد خرید اضافه شد', 'data' => $this->getLatestOrderWithItems()];
        } catch (\Exception $e) {
            handelError('error', $e);
            DB::rollBack();
            return ['status' => 500, 'message' => 'عملیات با شکست روبرو شد ', 'data' => null];
        }
    }


    public function updateAllCartOrder($data)
    {
        //data = ['variety_id','count','extraIds=1-2-3-4-5']
        if (isset($filter['date']) and !empty($filter['date']))
            $time = $filter['date'];
        else
            $time = Carbon::now();

        $varietyRep = app(VarietyRepository::class);
        $variety = $varietyRep->getByIdWithProduct($data['variety_id']);
        if ($variety->count < 1)
            return ['data' => [], 'message' => 'موجودی محصول به اتمام رسیده است', 'status' => 400];

        if (isset($data['guest_token']) and $data['guest_token'] != null)
            $this->guest_token = $userId = $data['guest_token'];
        else
            $userId = auth()->id();
        $orderActive = $this->getLatestOrderUser($userId);

        if (isset($data['extraIds'])) {
            $extras = $this->getExtras($data['extraIds']);
            $varietyCart = $this->checkProductOrder($orderActive, $data['variety_id'], $extras->pluck('id')->implode('-') ?? null);
        } else
            $varietyCart = $this->checkProductOrder($orderActive, $data['variety_id'], null);


        if ($orderActive->branch_id == null) {
            $branch = $varietyRep->getBranch($data['variety_id'])->id ?? null;
            $orderActive->update(['branch_id' => $branch]);
        }

        if (!$varietyCart) {
            $cart = OrderCart::create([
                'order_id' => $orderActive->id,
                'variety_id' => $data['variety_id'],
                'count' => $data['count'],
                'extras' => (isset($extras)) ? json_encode($extras) : null,
                'extra_ids' => (isset($extras)) ? $extras->pluck('id')->implode('-') : null,
                'price' => $variety->price,
                'discount' => $variety->discount
            ]);

        } else {
            $varietyCart->update(['count' => $data['count']]);
        }

        return $this->getLatestOrderWithItems();
    }

    public function minesCartOrder($data)
    {
        //data = ['variety_id','count','extraIds=1-2-3-4-5']
        DB::beginTransaction();
        try {
            if (isset($data['guest_token']) and $data['guest_token'] != null)
                $this->guest_token = $data['guest_token'];
            $orderActive = $this->getLatestOrderUser();
            if (isset($data['extraIds']) and $data['extraIds'] != "") {
                $extras = $this->getExtras($data['extraIds']);
                $varietyCart = $this->checkProductOrder($orderActive, $data['variety_id'], $extras->pluck('id')->implode('-') ?? null);
            } else
                $varietyCart = $this->checkProductOrder($orderActive, $data['variety_id'], null);

            if ($varietyCart) {
                if ($varietyCart->count - $data['count'] == 0) {
                    $varietyCart->delete();
                    $this->checkResetOrder($orderActive);
                } else
                    $varietyCart->update(['count' => $varietyCart->count - $data['count']]);
            }
            if (auth()->check()) {
                $check = app(VarietyRepository::class)->incrementCount($varietyCart->variety_id, $data['count']);
                if (!$check) {
                    DB::rollBack();
                    return ['status' => 400, 'message' => 'کم کردن از سبد خرید با مشکل مواجه شد', 'data' => []];
                }
            }
            return ['status' => 200, 'message' => 'تعداد محصول با موفقیت کم شد', 'data' => $this->getLatestOrderWithItems()];
        } catch (\Exception $e) {
            handelError('error', $e->getMessage());
            DB::rollBack();
            return ['status' => 500, 'message' => 'عملیات با شکست روبرو شد ', 'data' => null];
        }
    }

    public function checkResetOrder($order)
    {
        $count = count($order->orderCart);
        if ($count == 0) {
            $order->update(['branch_id' => null, 'delivery_time' => null]);
        }
    }

    public function deleteCartOrder($data)
    {
        DB::beginTransaction();
        try {
            //data = ['variety_id','count','extraIds=1-2-3-4-5']
            if (isset($data['guest_token']) and $data['guest_token'] != null)
                $this->guest_token = $userId = $data['guest_token'];
            else
                $userId = auth()->id();

            $orderActive = $this->getLatestOrderUser($userId);

            $extras = $this->getExtras($data['extraIds']);

            $varietyCart = $this->checkProductOrder($orderActive, $data['variety_id'], $extras->pluck('id')->implode('-') ?? null);
            if ($varietyCart) {
                $count = $varietyCart->count;
                $varietyCart->delete();
                if (auth()->check()) {
                    $check = app(VarietyRepository::class)->incrementCount($varietyCart->variety_id, $count);
                    if (!$check) {
                        DB::rollBack();
                        return response()->json(['status' => 400, 'message' => 'حذف محصول از سبد خرید با مشکل مواجه شد', 'data' => []], 400);
                    }
                }
            }
            return response()->json(['status' => 200, 'message' => 'محصول با موفقیت حذف شد', 'data' => $this->getLatestOrderWithItems()], 200);
        } catch (\Exception $e) {
            handelError('error', $e->getMessage());
            DB::rollBack();
            return ['status' => 500, 'message' => 'عملیات با شکست روبرو شد ', 'data' => null];
        }
    }


    public function getLatestOrderWithItems($token = null,$deliveryTypeId = null,$areaResult = null,$address_id = null)
    {
        if ($token != null)
            $this->guest_token = $token;

        if (!auth()->check())
            $order = Order::where('payment_status', PaymentStatus::FALSE)->where('order_status', OrderStatus::ORDERING)->where('guest_token', $this->guest_token)
                ->with('orderCart', 'branch:id,title,address,logo,min_order,vendor_id,lat,lng,cash_back_status,cash_back_percent,cash_back_max_price', 'address_order:id,user_id,address,plaque,postal_code,unit,lat,lng', 'deliveryType')->latest()->first();
        else {
            $order = Order::where('payment_status', PaymentStatus::FALSE)->where('order_status', OrderStatus::ORDERING)->where('user_id', auth()->id())
                ->with('orderCart', 'branch:id,title,address,logo,min_order,vendor_id,lat,lng,cash_back_status,cash_back_percent,cash_back_max_price', 'address_order:id,user_id,address,plaque,postal_code,unit,lat,lng', 'deliveryType')->latest()->first();
        }
        $address = null;
        if ($address_id){
            $address = $this->addressRepository->find($address_id);
        }

        return [
            'guest_token' => $this->guest_token,
            'order' => $order,
            'price' => $this->sumPriceOrder($order,$deliveryTypeId, $areaResult),
            'address' => $address
        ];
    }

    public function getCountItem()
    {
        $model = $this->model();
        $order = $this->getLatestOrderUser();
        return $order->orderCart->count() ?? 0;
    }

    public function checkProductOrder($order, $varietyId, $extra)
    {
        return OrderCart::where('order_id', $order->id)->where('variety_id', $varietyId)->where('extra_ids', $extra)->first();
    }

    public function getExtras($extraIds)
    {
        $extraIds = explode('-', $extraIds);
        return Extra::whereIn('id', $extraIds)->select('id', 'title', 'price')->get();
    }

    public function getLatestOrderUser($userId = null)
    {
        if ($userId == null)
            $userId = auth()->id();

        if ($userId == null)
            $order = Order::where('payment_status', PaymentStatus::FALSE)->where('order_status', OrderStatus::ORDERING)->where('guest_token', $this->guest_token)->latest()->first();
        else
            $order = Order::where('payment_status', PaymentStatus::FALSE)->where('order_status', OrderStatus::ORDERING)->where('user_id', $userId)->latest()->first();
        if (!$order)
            $order = $this->createOrderUser();
        return $order;
    }

    public function createOrderUser()
    {
        $order = new Order();
        $order->user_id = auth()->id();
        $order->guest_token = $this->guest_token;
        $order->save();
        return $order;
    }

    public function sumPriceOrder($order, $deliveryTypeId = null, $areaResult = null)
    {
        if (!$order instanceof Order) {
            $order = Order::find($order);
        }
        $sumProduct = 0;
        $sumExtras = 0;
        $sumPaking = 0;
        $sumDiscount = 0;

        if ($order == null)
            return [
                'sumProduct' => 0,
                'sumExtras' => 0,
                'sumPaking' => 0,
                'priceShippingOrder' => 0,
                'sumTotal' => 0,
                'totalPayment' => 0,
                'cashBack' => 0,
                'sumTax' => 0
            ];

        if ($order->orderCart != null || count($order->orderCart) != 0) {

            foreach ($order->orderCart as $item) {
                $extras = json_decode($item->extras);
                if ($item->discount != 0) {
                    $sumDiscountItem = PriceCalculation($item->price, $item->discount);
                    $sumProduct += PriceCalculation($item->price, $item->discount) * $item->count;
                    $sumDiscount += $sumDiscountItem;
                } else {
                    $sumProduct += $item->price * $item->count;
                }
                $sumExtras += collect($extras)->sum('price') * $item->count;
                $sumPaking += $item->variety->price_paking * $item->count;
            }
        }

        if ($sumProduct == 0) {
            $priceShippingOrder = 0;
            $sumTotal = 0;
            $sumTotalWithoutShippingPrice = 0;
            $tax = 0;
        } else {
            $priceShippingOrder = 0;

            // Check if delivery method is 'serve' or 'actual' - no shipping cost needed
            $deliveryMethod = $order->deliveryType?->delivery_method;
            if (in_array($deliveryMethod, ['serve', 'actual'])) {
                $priceShippingOrder = 0;
            } else {
                //$configDelivery=app(SettingRepository::class)->getSettingShippingType();
                $configDelivery=Setting::where('name', 'shippingType')->where('branch_id', null)->latest()->first()->toArray();
                Log::info($configDelivery);
                    if(isset($configDelivery['name']) and $configDelivery['value']=="withArea"){
                        $priceShippingOrder=$this->shipmentPriceWithType($deliveryMethod,$areaResult,$sumProduct);
                        Log::info($priceShippingOrder);

                    }
                    else if(isset($configDelivery['name']) and $configDelivery['value']=="withKilometer")
                    $priceShippingOrder=app(DeliveryRepository::class)->getCostDeliveryOrder($order->id,$order->deliveryType,$areaResult)['data']['cost']??0;
            }

                $sumTotalWithoutShippingPrice = $sumProduct + $sumExtras + $sumPaking;
                $sumTotal = $priceShippingOrder + $sumProduct + $sumExtras + $sumPaking;
                $tax = number_format($this->calculateTax($sumTotal) - $sumTotal, 0, '.', '');
            }

        $cashBack = 0;
        $sumTotalP = $sumTotal + $tax;
        $userBranchWallet = UserBranchWallet::where('user_id', $order->user_id)->where('branch_id', $order->branch_id)->first();
        if ($userBranchWallet != null and $userBranchWallet->amount > 0 and $userBranchWallet->expired_at > Carbon::now()) {
            if ($userBranchWallet->amount > $sumTotalWithoutShippingPrice)
                $cashBack = $sumTotalWithoutShippingPrice;
            else
                $cashBack = $userBranchWallet->amount;
        }
        return [
            'sumProduct' => $sumProduct,
            'sumExtras' => $sumExtras,
            'sumPaking' => $sumPaking,
            'priceShippingOrder' => $priceShippingOrder,
            'deliveryPrice' => $order?->delivery_price,
            'sumTotal' => $sumTotalP,
            'totalPayment' => $sumTotalP - $cashBack,
            'cashBack' => $cashBack,
            'sumTax' => $tax
        ];
    }


    public function shipmentPriceWithType($deliveryTypeMethod, $areaResult, $sumProduct = 0)
    {


        $priceShippingOrder = 0;

        if($deliveryTypeMethod == "" || $deliveryTypeMethod == null)
        {
            return $priceShippingOrder;
        }

        // If delivery method is 'serve' or 'actual', no shipping cost
        if (in_array($deliveryTypeMethod, ['serve', 'actual'])) {
            return 0;
        }

        Log::info("0000111111111");



        Log::info($areaResult);
        Log::info("0000111111111");
        //515
        if($areaResult && $areaResult['free_delivery'] == 1 && $areaResult['free_delivery_amount'] > $sumProduct){
            $priceShippingOrder = 0;
        }else{
            if($areaResult && $areaResult['free_delivery'] == 1 && $areaResult['free_delivery_amount'] > $sumProduct){
                $priceShippingOrder = 0;
            }elseif($areaResult){
                // Get delivery cost based on order amount
                if (isset($areaResult['order_amounts']) && is_array($areaResult['order_amounts'])) {
                    foreach ($areaResult['order_amounts'] as $orderAmount) {
                        if (isset($orderAmount['orderAmount1']) && isset($orderAmount['orderAmountDelivery'])) {
                            $amount1 = (int)$orderAmount['orderAmount1'];
                            $amount2 = (int)$orderAmount['orderAmount2'];
                            $deliveryCost = (int)$orderAmount['orderAmountDelivery'];

                            if ($sumProduct >= $amount1 && ($sumProduct <= $amount2 || $amount2 == null)) {
                                $priceShippingOrder = $deliveryCost;
                                break;
                            }
                        }
                    }
                }
            }
        }

        return $priceShippingOrder;
    }

    public function checkOrderUserAddressAndBranch($addressId, $BranchId)
    {
    }

    function calculateTax($baseAmount)
    {
        $setting = app(\App\Repositories\Setting\SettingRepository::class);
        $taxRate = $setting->getTaxPercent();
        // Calculate the tax amount
        $taxAmount = $baseAmount * ($taxRate / 100);
        // Calculate the total price
        $totalPrice = $baseAmount + $taxAmount;
        // Return an array with the calculated values
        return $totalPrice;
    }

    public function setAddress($addressId)
    {
        $order = $this->getLatestOrderUser();
        if ($order->branch_id == null || $order->orderCart == null)
            return ['status' => 400, 'message' => 'آدرس نادرست است', 'data' => null];

        if ($order['data']['check'] == false)
            return ['status' => 400, 'message' => $order['data']['message'], 'data' => null];
        $order->update(['address_id' => $addressId]);
        return ['status' => 200, 'message' => $order['data']['message'] ?? 'آدرس با موفقیت ثبت شد', 'data' => $this->getLatestOrderWithItems()];
    }

    public function getListAddressAccessCart($request)
    {
        $order = $this->getLatestOrderUser();
        $dataSettings = app(SettingRepository::class)->getSettingWithNames(['systemMode', 'maxDistance']);
        $modeSystem = $dataSettings['systemMode'] ?? "single";
        $maxDistance = $dataSettings['maxDistance'] ?? 10;
        $branch = $order->branch;

        $query = Address::where('user_id', auth()->id());

        if ($request->has('address_title') && !empty($request->address_title)) {
            $search = $request->address_title;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'LIKE', "%{$search}%")
                  ->orWhere('address', 'LIKE', "%{$search}%")
                  ->orWhere('postal_code', 'LIKE', "%{$search}%")
                  ->orWhere('description', 'LIKE', "%{$search}%");
            });
        }
        if ($request->has('address_area') && !empty($request->address_area)) {
            $query->where('area_id', $request->address_area);
        }

        $address = $query->where(function ($query) use ($branch, $modeSystem, $maxDistance) {
            // if ($branch->lat != null and $branch->lng != null and $modeSystem != 'single') {
            //     $lat = $branch->lat;
            //     $lng = $branch->lng;
            //     $radius = $maxDistance;
            //     $haversine = "(6371 * acos(cos(radians($lat)) * cos(radians(lat)) * cos(radians(lng) - radians($lng)) + sin(radians($lat)) * sin(radians(lat))))";
            //     $query->selectRaw("{$haversine} AS distance")
            //         ->whereRaw("{$haversine} < ?", [$radius]);
            // }
        })->get();

        return $address ?? [];
    }

    public function orderListBefore()
    {
        $order = Order::where('payment_status', PaymentStatus::TRUE)->where('user_id', auth()->id())->with('orderCart', 'branch:id,title,address,logo,min_order,vendor_id')->latest()->first();
        return $order ?? [];
    }

    public function OrderListDeleteCart()
    {
        $order = Order::where('payment_status', PaymentStatus::FALSE)->where('order_status', OrderStatus::ORDERING)->where('user_id', auth()->id())->latest()->first();
        if ($order != null)
            $order->orderCart()->delete();
    }

    public function getListGateways()
    {
        return Gateway::where('status', \App\Enums\Gateway::TRUE)->get();
    }

    public function checkCoupon($data)
    {
        $dateCarbon = Carbon::now();
        $coupon = Coupon::where('code', $data['coupon'])->where('active_from', '<=', $dateCarbon)->where('active_to', '>=', $dateCarbon)->first();
        if ($coupon == null)
            return ['status' => 400, 'message' => 'کد تخفیفی یافت نشد'];
        $order = $this->getLatestOrderWithItems();
        if ($coupon['branch_id'] != 0 and $order['order']['branch_id'] != $coupon['branch_id'])
            return ['status' => 400, 'message' => 'کد تخفیفی برای این آشپز نیست'];
        $price = $order['price'];
        $sumTotal = $price['sumTotal'];
        if ($coupon->min_price_order != 0 and $coupon->min_price_order > $price['sumProduct'])
            return ['status' => 400, 'message' => 'حداقل سفارش برای استفاده از کد تخفیف ' . $coupon['min_price_order'] . ' میباشد'];
        if ($coupon['type'] == 1) {
            $s = PriceCalculation($sumTotal, $coupon['percent']);
            $s = $sumTotal - $s;
            if ($s > $coupon['max_price_discount'])
                $s = $coupon['max_price_discount'];
            $fPrice = $price['sumTotal'] - $s;
            return ['status' => 200, 'data' => ['discountPrice' => (int)$s, 'coupon' => $coupon]];
        } else {
            $fPrice = $price['sumTotal'] - $coupon['price'];
            return ['status' => 200, 'data' => ['discountPrice' => (int)$coupon['price'], 'coupon' => $coupon]];
        }
    }

    public function createComment($data, $id)
    {
        $order = $this->showWithUser($id);
        $checkComment = app(CommentRepository::class)->checkCommentUserOrder($id);
        if ($checkComment != null)
            return ['status' => 500, "message" => "شما قبلا نظر خود را ثبت کرده اید"];
        if ($order) {
            $products = collect($order->carts)->toArray();
            $varietyIds = [];
            $varietyNames = [];
            foreach ($products as $product) {
                $varietyIds[] = $product['variety']['id'];
                $varietyNames[$product['variety']['id']]['title'] = $product['variety']['title'];
                $varietyNames[$product['variety']['id']]['product_id'] = $product['variety']['product_id'];
            }
            $rates = [];
            $count = 0;
            $rateSum = 0;
            foreach ($data['rates'] as $item) {
                if (in_array($item['id'], $varietyIds)) {
                    $count += 1;
                    $rateSum += $item['rate'];
                    $rates[] = [
                        "variety_id" => $item['id'],
                        "id" => $varietyNames[$item['id']]['product_id'],
                        "title" => $varietyNames[$item['id']]['title'],
                        "rate" => $item['rate'],
                    ];
                }
            }
            $data = [
                'user_id' => auth()->id(),
                'rate' => ($rateSum != 0 and $count != 0) ? $rateSum / $count : 0,
                'products' => json_encode($rates),
                'text' => $data['text'],
                'branch_id' => $order->branch_id,
                'order_id' => $order->id,
            ];
            $comment = app(CommentRepository::class)->store($data);
            if ($comment) {
                return ['status' => 200, "data" => $comment];
            }
        }
        return ['status' => 500, "message" => "در ثبت اطلاعات خطایی رخ داده است"];
    }

    public function setDeliveryOrder($orderId)
    {
        $user = auth()->user();
        if (!$user->role_id == 5)
            return ['status' => 403, "message" => "شما دسترسی برای قبول ارسال ندارید"];
        $order = $this->model->where('id', $orderId)->where('order_status', OrderStatus::READYTOSEND)->first();
        if ($order == null) {
            return ['status' => 403, "message" => "سفارشی یافت نشد"];
        } else {
            $this->model->where('id', $orderId)->where('order_status', OrderStatus::READYTOSEND)->update([
                'driver_id' => auth()->id(),
                'order_status' => OrderStatus::SENDING,
            ]);
            return ['status' => 200, "message" => "ارسال سفارش برای شما ست شد"];
        }
    }

    public function updateGustOrderUserId($userId, $token)
    {

        try {
            $order = $this->model->where('guest_token', $token)->with('OrderCart')->first();

            $deletes = [];
            foreach ($order->orderCart as $item) {
//            if ($item->variety->count == 0 || $item->variety->count < $item->count)
//                $item->delete();
                $check = app(VarietyRepository::class)->decrementCount($item->variety_id, $item->count);
                if (!$check) {
                    $deletes[] = $item;
                    $item->delete();
                }
            }
            $order->update(['user_id' => $userId]);
            return ['data' => ['deletes' => $deletes], 'message' => '', 'status' => 200];
        } catch (\Exception $e) {
            return ['data' => [], 'message' => $e->getMessage(), 'status' => 500];
        }
    }

    function getNextDateForDay(string $weekDay, $time): string
    {
        $today = Carbon::today();

        if ($today->englishDayOfWeek === ucfirst($weekDay)) {
            $date = $today;
        } else {
            $date = $today->copy()->modify("next $weekDay");
        }
        $dateTime = $date->toDateString() . ' ' . $time;

        $carbonDateTime = Carbon::parse($dateTime);

        return Jalalian::fromCarbon($carbonDateTime)->format('Y/m/d H:i');
    }

    public function updateOrderAndCartsWithHistory($orderId, array $orderData, $diff = 0)
    {
        DB::beginTransaction();
        try {
            // دریافت سفارش موجود
            $order = $this->model->findOrFail($orderId);

            // گرفتن وضعیت فعلی سفارش
            $oldOrderData = $order->toArray();
            // گرفتن وضعیت فعلی سبد خرید مرتبط
            $oldOrderCarts = \App\Models\OrderCart::where('order_id', $orderId)->get()->toArray();

            $newOrder = $oldOrderData;
            $newOrder['order_price'] = $this->computeOrder($orderData,$oldOrderData);

            // ثبت تاریخچه سفارش شامل اطلاعات سفارش و سبد خرید
            \App\Models\OrderHistory::create([
                'order_id' => $orderId,
                'old_data' => json_encode([
                    'order' => $oldOrderData,
                    'order_carts' => $oldOrderCarts,
                ]),
            ]);

            /**
             * به‌روزرسانی سبد خرید:
             * فرض بر این است که آرایه‌ی $cartData شامل آیتم‌هایی است که ممکن است دارای کلید 'id' باشند.
             * اگر 'id' وجود داشته باشد، آن آیتم به‌روزرسانی می‌شود؛ در غیر این صورت به عنوان آیتم جدید وارد می‌شود.
             */

            $newOrderCart = [];
            foreach ($orderData as $cartItem) {
                if (isset($cartItem['id'])) {
                    // به‌روزرسانی آیتم موجود
                    $orderCart = \App\Models\OrderCart::with('variety')->find($cartItem['id']);

                    if ($orderCart && $orderCart->order_id == $orderId) {
                        $orderCart->update($cartItem);
                        array_push($newOrderCart, $orderCart);
                    }
                } else {
                    // ایجاد آیتم جدید در سبد خرید
                    $cartItem['order_id'] = $orderId;
                    $cartItem['variety_id'] = $cartItem['variety']['id'];
                    $cartItem['price'] = $cartItem['variety']['price'];
                    $cartItem['discount'] = $cartItem['variety']['discount'];
                    $create = \App\Models\OrderCart::create($cartItem);
                    $create->load('variety');
                    array_push($newOrderCart, $create);
                }
            }
            if(count($newOrderCart) > 0){
                $newOrder['carts'] = json_encode($newOrderCart);
                // به‌روزرسانی اطلاعات سفارش
                if($diff > 0){
                    $newOrder['order_status'] = OrderStatus::WATINGPAYMENT;
                    $newOrder['payment_status'] = PaymentStatus::FALSE;
                }
                $order->update($newOrder);
            }else{
                return ['status' => 500, 'message' => 'خطا در به‌روزرسانی سفارش', 'data' => []];
            }

            DB::commit();
            return ['status' => 200, 'message' => 'سفارش و سبد خرید با موفقیت به‌روزرسانی شدند', 'data' => []];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return ['status' => 500, 'message' => 'خطا در به‌روزرسانی سفارش', 'data' => []];
        }
    }

    public function computeOrder($orderData,$oldOrderData)
    {
        $finalPrice = 0;

        foreach ($orderData as $orderItem) {
            if(is_array($orderItem)){
                if (isset($orderItem['id'])) {
                    $finalPrice += $orderItem['price'] * $orderItem['count'];
                } else {
                    $finalPrice += $orderItem['variety']['price'] * $orderItem['count'];
                }
            }
        }
        $finalPrice = $oldOrderData['tax_price'] + $finalPrice;

        return $finalPrice;
    }

    /**
     * حذف سفارش باز (در سبد خرید) کاربر
     */
    public function deleteUserActiveOrder($userId)
    {
        DB::beginTransaction();
        try {
            // پیدا کردن سفارش‌های باز کاربر (ordering یا wating_payment)
            $orders = Order::where('user_id', $userId)
                ->whereIn('order_status', [OrderStatus::ORDERING->value, OrderStatus::WATINGPAYMENT->value])
                ->with('orderCart')
                ->get();

            if ($orders->isEmpty()) {
                return [
                    'status' => 404,
                    'message' => 'سفارش بازی برای این کاربر یافت نشد',
                    'data' => null
                ];
            }

            $varietyRepository = app(VarietyRepository::class);
            $deletedCount = 0;

            foreach ($orders as $order) {
                // بازگرداندن موجودی محصولات به انبار
                foreach ($order->orderCart as $item) {
                    $variety = Variety::find($item->variety_id);
                    if ($variety) {
                        $variety->increment('count', $item->count);
                    }
                }

                // حذف سفارش
                $order->delete();
                $deletedCount++;
            }

            DB::commit();

            return [
                'status' => 200,
                'message' => "{$deletedCount} سفارش با موفقیت حذف شد",
                'data' => ['deleted_count' => $deletedCount]
            ];

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('خطا در حذف سفارش کاربر: ' . $e->getMessage());
            return [
                'status' => 500,
                'message' => 'خطا در حذف سفارش',
                'data' => null
            ];
        }
    }
}
